﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nn/util/util_BitFlagSet.h>

#include <nn/hid/system/hid_InputDetection.h>

#include <nn/idle/idle_Result.h>
#include <nn/idle/detail/idle_Log.h>
#include <nn/idle/server/idle_HandlerManager.h>

#include "idle_InactivityCounter.h"
#include "idle_ContextDependentGoverner.h"
#include "idle_BaseClock.h"
#include "idle_HandlerImpl.h"
#include "idle_PolicyParamHolder.h"

namespace nn { namespace idle { namespace server {

namespace {

    // NOTE: すべて単一のサーバスレッドから呼び出しているので、ミューテックス等での同期は不要
    // 複数スレッドから呼ばれうる場合、各ハンドラはスレッドセーフ性を意識しなくて良いようにここで保護するのが良い

    InactivityCounter g_InactivityCounter;

    ContextDependentGoverner g_ContextDependentGoverner;

    BaseClock g_BaseClock;

    HandlerImpl g_HandlerImpl;

    PolicyParamHolder g_PolicyParamHolder;

    nn::TimeSpan GetLastInputDetectedTime(const nn::hid::system::InputSourceIdSet& enabledInputSourceIdSet) NN_NOEXCEPT
    {
        // 最後の確認以降アクティビティがあったかを調べる
        nn::hid::system::InputSourceState inputState;
        nn::hid::system::GetInputSourceState(&inputState, enabledInputSourceIdSet); // すべての入力を対象にする
        auto detectedTime = nn::os::ConvertToTimeSpan(nn::os::Tick(inputState.timestamp));
        // NN_DETAIL_IDLE_TRACE("timestamp %lld usec\n", detectedTime.GetMicroSeconds());
        return detectedTime;
    }
}

void InitializeHandlerManager() NN_NOEXCEPT
{
    g_BaseClock.Initialize();
    g_HandlerImpl.Initialize();
    g_InactivityCounter.Initialize(&g_HandlerImpl, g_BaseClock.GetNow());
    g_ContextDependentGoverner.Initialize(&g_InactivityCounter);
    g_PolicyParamHolder.Initialize();

    nn::hid::system::InitializeInputDetector();
    LoadAndApplySettings();

    EnableHandlers();
}

void EnableHandlers() NN_NOEXCEPT
{
    g_HandlerImpl.Enable();
    ResetIdleTimeCount(); // 中で NotifyActivity が呼ばれる
}

void DisableHandlers() NN_NOEXCEPT
{
    g_HandlerImpl.Disable();
}

void LoadAndApplySettings() NN_NOEXCEPT
{
    g_PolicyParamHolder.LoadFirmwareDebugSettings();
    g_PolicyParamHolder.LoadSystemSettings();

    // 本体設定を再適用するときには必ず再カウント
    ResetIdleTimeCount();

    g_HandlerImpl.UpdatePolicy(g_PolicyParamHolder.Get());
}

Result SetHandlingContext(const HandlingContext& handlingContext) NN_NOEXCEPT
{
    // 値チェック
    if ( handlingContext.overrideAutoSleepTimeInConsoleInSeconds < 0 ||
         handlingContext.overrideAutoSleepTimeInHandheldInSeconds < 0 ||
         handlingContext.overrideDimmingTimeInConsoleInSeconds < 0 ||
         handlingContext.overrideDimmingTimeInHandheldInSeconds < 0 ||
         handlingContext.overrideAutoSleepTimeInConsoleInSeconds >= MaxOverrideAutoSleepTimeInSeconds ||
         handlingContext.overrideAutoSleepTimeInHandheldInSeconds >= MaxOverrideAutoSleepTimeInSeconds ||
         handlingContext.overrideDimmingTimeInConsoleInSeconds >= MaxOverrideDimmingTimeInSeconds ||
         handlingContext.overrideDimmingTimeInHandheldInSeconds >= MaxOverrideDimmingTimeInSeconds )
    {
        NN_RESULT_THROW(nn::idle::ResultInvalidTime());
    }

    // 変更内容によってカウンタのリセットの要不要を判断
    HandlingContext newHandlingContext;
    g_ContextDependentGoverner.InputHandlingContextChange(&newHandlingContext, g_BaseClock.GetNow(), g_PolicyParamHolder.Get().handlingContext, handlingContext);
    g_PolicyParamHolder.UpdateHandlingContext(newHandlingContext);
    g_HandlerImpl.UpdatePolicy(g_PolicyParamHolder.Get());

    NN_RESULT_SUCCESS;
}

void ResetIdleTimeCount() NN_NOEXCEPT
{
    auto nowTime = g_BaseClock.GetNow();
    auto wasNewInputDetected = g_InactivityCounter.InputLastActiveTime(nowTime);
    auto context = g_PolicyParamHolder.Get().handlingContext;
    if ( g_ContextDependentGoverner.UpdateContext(&context, nowTime, wasNewInputDetected) )
    {
        g_PolicyParamHolder.UpdateHandlingContext(context);
        g_HandlerImpl.UpdatePolicy(g_PolicyParamHolder.Get());
    }
}

void UpdateHandlerStatus(nn::TimeSpan interval) NN_NOEXCEPT
{
    // 仮想環境では注入された interval 分だけ現在時刻が経過するようにする
    g_BaseClock.Proceed(interval);

    // hid の InputDetector から最後の操作時刻を取得して InactivityCounter に注入する
    // 最終操作時刻が更新されれば NotifyActivity され、true が返る
    auto wasNewInputDetected = g_InactivityCounter.InputLastActiveTime(
        GetLastInputDetectedTime(g_PolicyParamHolder.Get().handlingContext.enabledInputSourceIdSet)
    );

    // ハンドリングコンテキストポリシーの更新
    auto nowTime = g_BaseClock.GetNow();
    auto context = g_PolicyParamHolder.Get().handlingContext;
    if ( g_ContextDependentGoverner.UpdateContext(&context, nowTime, wasNewInputDetected) )
    {
        g_PolicyParamHolder.UpdateHandlingContext(context);
        g_HandlerImpl.UpdatePolicy(g_PolicyParamHolder.Get());
    }

    // InactivityCounter から現在の無操作時間を教えてもらう
    auto elapsedIdleTime = g_InactivityCounter.GetInactiveTime(nowTime);
    // NN_DETAIL_IDLE_TRACE("elapsedIdleTime=%lld ms\n", elapsedIdleTime.GetMilliSeconds());
    g_HandlerImpl.NotifyInactivity(elapsedIdleTime);
}

nn::os::SystemEventType* GetAutoPowerDownEvent() NN_NOEXCEPT
{
    return g_HandlerImpl.GetAutoPowerDownEvent();
}

bool IsAutoPowerDownOn() NN_NOEXCEPT
{
    return g_HandlerImpl.IsAutoPowerDownOn();
}

bool IsDimmingOn() NN_NOEXCEPT
{
    return g_HandlerImpl.IsDimmingOn();
}

}}} // namespace nn::idle::server

