﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/sf/sf_NativeHandle.h> // for nn::sf::NativeHandle
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_Types.h> // for nn::sf::SharedPointer

#include <nn/os.h>
#include <nn/os/os_Mutex.h>

#include <nn/idle/idle_SystemApi.h>
#include <nn/idle/idle_ServiceName.h>
#include <nn/idle/idle_ShimInternal.h>
#include <nn/idle/detail/idle_IPolicyManagerSystem.sfdl.h>

namespace nn { namespace idle {

namespace {

const int NumberOfClients = 1;
sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
sf::ShimLibraryObjectHolder<detail::IPolicyManagerSystem> g_PolicyManagerSystemHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

nn::os::Mutex g_ManagerAccessMutex(false);

} // namespace

void InitializeForSystem() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_PolicyManagerSystemHolder, nn::idle::ServiceNameForSystem));
}

void FinalizeForSystem() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);
    g_PolicyManagerSystemHolder.FinalizeHolder();
}

void InitializeWith(sf::SharedPointer<detail::IPolicyManagerSystem>&& manager) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);
    g_PolicyManagerSystemHolder.InitializeHolderDirectly(std::move(manager));
}

nn::sf::SharedPointer<detail::IPolicyManagerSystem> GetPolicyManagerSystem() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);
    return g_PolicyManagerSystemHolder.GetObject();
}

Result GetAutoPowerDownEvent(nn::os::SystemEventType* pOutEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEvent);
    nn::sf::NativeHandle sfHandle;
    NN_RESULT_DO(g_PolicyManagerSystemHolder.GetObject()->GetAutoPowerDownEvent(&sfHandle));
    nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfHandle.GetOsHandle(), sfHandle.IsManaged(), eventClearMode);
    sfHandle.Detach();
    NN_RESULT_SUCCESS;
}

Result SetHandlingContext(const HandlingContext& handlingContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(handlingContext.overrideAutoSleepTimeInHandheldInSeconds, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(handlingContext.overrideAutoSleepTimeInConsoleInSeconds, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(handlingContext.overrideDimmingTimeInHandheldInSeconds, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(handlingContext.overrideDimmingTimeInConsoleInSeconds, 0);
    NN_SDK_REQUIRES_LESS(handlingContext.overrideAutoSleepTimeInHandheldInSeconds, MaxOverrideAutoSleepTimeInSeconds);
    NN_SDK_REQUIRES_LESS(handlingContext.overrideAutoSleepTimeInConsoleInSeconds, MaxOverrideAutoSleepTimeInSeconds);
    NN_SDK_REQUIRES_LESS(handlingContext.overrideDimmingTimeInHandheldInSeconds, MaxOverrideDimmingTimeInSeconds);
    NN_SDK_REQUIRES_LESS(handlingContext.overrideDimmingTimeInConsoleInSeconds, MaxOverrideDimmingTimeInSeconds);
    NN_RESULT_DO(g_PolicyManagerSystemHolder.GetObject()->SetHandlingContext(handlingContext));
    NN_RESULT_SUCCESS;
}

Result LoadAndApplySettings() NN_NOEXCEPT
{
    NN_RESULT_DO(g_PolicyManagerSystemHolder.GetObject()->LoadAndApplySettings());
    NN_RESULT_SUCCESS;
}

Result ReportUserIsActive() NN_NOEXCEPT
{
    NN_RESULT_DO(g_PolicyManagerSystemHolder.GetObject()->ReportUserIsActive());
    NN_RESULT_SUCCESS;
}

}} // namespace nn::idle
