﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/psc.h>

#include <nn/i2c/i2c_ServiceType.h>
#include <nn/i2c/i2c_IManager.sfdl.h>
#include <nn/i2c/driver/i2c_Suspend.h>
#include <nn/i2c/server/i2c_ManagerImpl.h>

#include "i2c_ServerUtil.h"

namespace nn { namespace i2c { namespace server {

namespace {

    // MyServerManager に渡すオプション用構造体
    struct MyServerManagerOption
    {
        // ポインタ転送に使用するバッファのサイズ
        // ExecuteCommandList() の List の最大値がこれになる
        static const size_t PointerTransferBufferSize = 1024;
    };

    // TORIAEZU : 多めに確保 (I2C セッションの数: driver 側で想定するセッション数 + クライアントプロセスの数:10)
    const int ClientNum = 10;
    const int SessionCountMax = nn::i2c::MaxDriverSessions + ClientNum;
    class I2cServerManager
        : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, 1, MyServerManagerOption>
    {};
    I2cServerManager g_ServerManager;
    nn::sf::UnmanagedServiceObject<nn::i2c::IManager, nn::i2c::server::ManagerImpl> g_ServiceObject;
    nn::psc::PmModule g_PmModule;

    bool g_IsHipcServerRegistered = false;
    bool g_IsHipcServerStarted = false;

    // TORIAEZU : Pcv で使うセッション数。多めに確保
    const int SessionCountMaxPowerBus = 20;
    class I2cServerManagerPowerBus
        : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMaxPowerBus, 1, MyServerManagerOption>
    {};
    I2cServerManagerPowerBus g_ServerManagerPowerBus;
    nn::sf::UnmanagedServiceObject<nn::i2c::IManager, nn::i2c::server::ManagerImpl> g_ServiceObjectPowerBus;
    nn::psc::PmModule g_PmModulePowerBus;
}

nn::sf::SharedPointer<nn::i2c::IManager> GetServiceObject() NN_NOEXCEPT
{
    return g_ServiceObject.GetShared();
}

nn::sf::SharedPointer<nn::i2c::IManager> GetServiceObjectPowerBus() NN_NOEXCEPT
{
    return g_ServiceObjectPowerBus.GetShared();
}

void RegisterHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_IsHipcServerRegistered);

    // サービス名の登録とポートの初期化
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager.RegisterObjectForPort(g_ServiceObject.GetShared(), SessionCountMax, nn::i2c::I2cServiceName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManagerPowerBus.RegisterObjectForPort(g_ServiceObjectPowerBus.GetShared(), SessionCountMaxPowerBus, nn::i2c::I2cPowerBusServiceName));

    g_IsHipcServerRegistered = true;
}

void StartHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(!g_IsHipcServerStarted);

    // PSC へのモジュール登録
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_PmModule.Initialize(
            nn::psc::PmModuleId_I2c,
            I2cDependencies,
            sizeof(I2cDependencies) / sizeof(I2cDependencies[0]),
            nn::os::EventClearMode_ManualClear)
    );
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_PmModulePowerBus.Initialize(
            nn::psc::PmModuleId_I2cPowerBus,
            I2cPowerBusDependencies,
            sizeof(I2cPowerBusDependencies) / sizeof(I2cPowerBusDependencies[0]),
            nn::os::EventClearMode_ManualClear)
    );

    g_ServerManager.Start();
    g_ServerManagerPowerBus.Start();

    g_IsHipcServerStarted = true;
}

void LoopHipcServer() NN_NOEXCEPT
{
    nn::psc::PmState    state         = nn::psc::PmState_Unknown;
    nn::psc::PmState    previousState = nn::psc::PmState_Unknown;
    nn::psc::PmFlagSet  flags;
    static const ServerThreadParam<I2cServerManager> threadParam =
    {
        &g_PmModule,
        &g_ServerManager,
        StateEdge_MinimumAwakeToSleepReady
    };

    bool ipcEnable = true;

    while (NN_STATIC_CONDITION(true))
    {
        if (ipcEnable)
        {
            // IPC 許可状態
            IpcLoop<I2cServerManager, nn::psc::PmModuleId_I2c>(&previousState, &state, &flags, &threadParam);
        }
        else
        {
            // IPC 禁止状態 (PSC のイベントのみ待ち受ける)
            g_PmModule.GetEventPointer()->Wait();
            g_PmModule.GetEventPointer()->Clear();

            NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));
        }

        auto stateEdge = GetStateEdge(previousState, state);
        switch (stateEdge)
        {
        case StateEdge_MinimumAwakeToSleepReady:
            NN_SDK_LOG("[bus] I2c suspend\n");
            nn::i2c::driver::SuspendBuses();
            ipcEnable = false;
            break;

        case StateEdge_EssentialServicesAwakeToMinimumAwake:
            NN_SDK_LOG("[bus] I2c resume\n");
            nn::i2c::driver::ResumeBuses();
            ipcEnable = true;
            break;

        case StateEdge_SleepReadyToEssentialServicesSleepReady:
        case StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake:
            ipcEnable = false;
            break;

        default:
            ipcEnable = true;
            break;
        }

        g_PmModule.Acknowledge(state, nn::ResultSuccess());
        previousState = state;
    }
}

void LoopHipcServerPowerBus() NN_NOEXCEPT
{
    nn::psc::PmState    state         = nn::psc::PmState_Unknown;
    nn::psc::PmState    previousState = nn::psc::PmState_Unknown;
    nn::psc::PmFlagSet  flags;
    static const ServerThreadParam<I2cServerManagerPowerBus> threadParam =
    {
        &g_PmModulePowerBus,
        &g_ServerManagerPowerBus,
        StateEdge_SleepReadyToEssentialServicesSleepReady
    };

    bool ipcEnable = true;

    while (NN_STATIC_CONDITION(true))
    {
        if (ipcEnable)
        {
            // IPC 許可状態
            IpcLoop<I2cServerManagerPowerBus, nn::psc::PmModuleId_I2cPowerBus>(&previousState, &state, &flags, &threadParam);
        }
        else
        {
            // IPC 禁止状態 (PSC のイベントのみ待ち受ける)
            g_PmModulePowerBus.GetEventPointer()->Wait();
            g_PmModulePowerBus.GetEventPointer()->Clear();

            NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModulePowerBus.GetRequest(&state, &flags));
        }

        auto stateEdge = GetStateEdge(previousState, state);
        switch (stateEdge)
        {
        case StateEdge_SleepReadyToEssentialServicesSleepReady:
            NN_SDK_LOG("[bus] I2c PowerBus suspend\n");
            nn::i2c::driver::SuspendPowerBuses();
            ipcEnable = false;
            break;

        case StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake:
            NN_SDK_LOG("[bus] I2c PowerBus resume\n");
            nn::i2c::driver::ResumePowerBuses();
            ipcEnable = true;
            break;

        default:
            ipcEnable = true;
            break;
        }

        g_PmModulePowerBus.Acknowledge(state, nn::ResultSuccess());
        previousState = state;
    }
}

void StopHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(g_IsHipcServerStarted);

    g_ServerManager.RequestStop();
    g_ServerManagerPowerBus.RequestStop();

    g_IsHipcServerStarted = false;
}

}}}
