﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/i2c/driver/i2c.h>
#include <nn/util/util_BitPack.h>

#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include "detail/i2c_CommandListFormat.h"
#include <nn/i2c/i2c_CommandListFormatter.h>

namespace nn {
namespace i2c {

CommandListFormatter::CommandListFormatter(void* pCommandList, size_t commandLength) NN_NOEXCEPT :
    m_CurrentIndex(0),
    m_CommandListLength(commandLength),
    m_pCommandList(pCommandList)
{
    NN_ABORT_UNLESS(commandLength <=  nn::i2c::CommandListLengthCountMax );
}

Result CommandListFormatter::IsEnqueueAble( size_t enqueueBytes ) const NN_NOEXCEPT
{
    size_t availableCmdLength = m_CommandListLength - m_CurrentIndex;
    if( availableCmdLength < enqueueBytes )
    {
        return ResultCommandListFull();
    }
    NN_RESULT_SUCCESS;
}

Result   CommandListFormatter::EnqueueReceiveCommand( nn::i2c::TransactionOption inOption, size_t dataBytes ) NN_NOEXCEPT
{
    const size_t EnqueueCommandLength = 2;
    NN_RESULT_DO( IsEnqueueAble( EnqueueCommandLength ) );

    uint8_t* pCommandList = static_cast<uint8_t*>(m_pCommandList);

    // byte1
    pCommandList[m_CurrentIndex]  = nn::i2c::detail::CommandId_Receive << nn::i2c::detail::CommonCommandFormat::CommandHeader::Pos;
    pCommandList[m_CurrentIndex] |= ( ( inOption & nn::i2c::TransactionOption_StartCondition ) ? 1 : 0 ) << nn::i2c::detail::ReceiveCommandFormat::StartCondition::Pos;
    pCommandList[m_CurrentIndex] |= ( ( inOption & nn::i2c::TransactionOption_StopCondition )  ? 1 : 0 ) << nn::i2c::detail::ReceiveCommandFormat::StopCondition::Pos;
    m_CurrentIndex++;

    // byte2
    pCommandList[m_CurrentIndex] = dataBytes << nn::i2c::detail::ReceiveCommandFormat::DataBytes::Pos;
    m_CurrentIndex++;
    NN_RESULT_SUCCESS;
}

Result   CommandListFormatter::EnqueueSendCommand( nn::i2c::TransactionOption inOption, const nn::Bit8* pInData, size_t dataBytes ) NN_NOEXCEPT
{
    const size_t EnqueueCommandLength = 2 + dataBytes;
    NN_RESULT_DO( IsEnqueueAble( EnqueueCommandLength ) );

    uint8_t* pCommandList = static_cast<uint8_t*>(m_pCommandList);

    // byte1
    pCommandList[m_CurrentIndex]  = nn::i2c::detail::CommandId_Send << nn::i2c::detail::CommonCommandFormat::CommandHeader::Pos;
    pCommandList[m_CurrentIndex] |= ( ( inOption & nn::i2c::TransactionOption_StartCondition ) ? 1 : 0 ) << nn::i2c::detail::SendCommandFormat::StartCondition::Pos;
    pCommandList[m_CurrentIndex] |= ( ( inOption & nn::i2c::TransactionOption_StopCondition )  ? 1 : 0 ) << nn::i2c::detail::SendCommandFormat::StopCondition::Pos;
    m_CurrentIndex++;

    // byte2
    pCommandList[m_CurrentIndex] = dataBytes << nn::i2c::detail::SendCommandFormat::DataBytes::Pos;
    m_CurrentIndex++;

    for(size_t i = 0; i < dataBytes; i++ )
    {
        pCommandList[m_CurrentIndex++] = pInData[i];
    }
    NN_RESULT_SUCCESS;
}

Result   CommandListFormatter::EnqueueSleepCommand( int sleepMicroSeconds ) NN_NOEXCEPT
{
    const size_t EnqueueCommandLength = 2;
    NN_RESULT_DO( IsEnqueueAble( EnqueueCommandLength ) );

    uint8_t* pCommandList = static_cast<uint8_t*>(m_pCommandList);

    // byte1
    pCommandList[m_CurrentIndex]  = nn::i2c::detail::CommandId_Extension << nn::i2c::detail::CommonCommandFormat::CommandHeader::Pos;
    pCommandList[m_CurrentIndex] |= nn::i2c::detail::SubCommandId::SubCommandId_Sleep << nn::i2c::detail::CommonCommandFormat::SubCommandHeader::Pos;
    m_CurrentIndex++;

    // byte2
    pCommandList[m_CurrentIndex]  = sleepMicroSeconds << nn::i2c::detail::SleepCommandFormat::SleepMicroSeconds::Pos;
    m_CurrentIndex++;
    NN_RESULT_SUCCESS;
}

} // i2c
} // nn
