﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>

#include <nn/i2c/detail/i2c_Log.h>
#include <nn/i2c/driver/detail/i2c_I2cSessionContext.h>
#include <nn/i2c/driver/i2c_II2cDriver.h>
#include <nn/i2c/driver/i2c_I2cDeviceProperty.h>
#include <nn/i2c/i2c_ResultForPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "../../detail/i2c_CommandListFormat.h"

namespace nn { namespace i2c { namespace driver { namespace detail {

I2cSessionContext::I2cSessionContext(int maxRetryCount, nn::TimeSpan retryInterval) NN_NOEXCEPT
    : m_MaxRetryCount(maxRetryCount)
    , m_RetryInterval(retryInterval)
{

}

I2cSessionContext::~I2cSessionContext() NN_NOEXCEPT
{
    Close();
}

nn::Result I2cSessionContext::Open(I2cDeviceProperty* pDevice, nn::ddsf::AccessMode accessMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);

    bool needsRollback = true;
    bool isFirstSessionOnDevice = !(pDevice->HasAnyOpenSession());

    // 発見したデバイスオブジェクトをセッションと相互に紐づけ
    NN_RESULT_DO(nn::ddsf::OpenSession(pDevice, this, accessMode));
    NN_UTIL_SCOPE_EXIT
    {
        if ( needsRollback )
        {
            nn::ddsf::CloseSession(this);
        }
    };

    auto& driver = pDevice->GetDriver().SafeCastTo<II2cDriver>();
    if ( isFirstSessionOnDevice )
    {
        NN_RESULT_DO(driver.InitializeDevice(pDevice));
    }
    NN_UTIL_SCOPE_EXIT
    {
        if ( needsRollback && isFirstSessionOnDevice )
        {
            driver.FinalizeDevice(pDevice);
        }
    };

    needsRollback = false; // 成功
    NN_RESULT_SUCCESS;
}

void I2cSessionContext::Close() NN_NOEXCEPT
{
    // 多重クローズは許容
    if ( !IsOpen() )
    {
        return;
    }

    // GetDevice() は Close 前でないと呼べないので注意
    auto& device = GetDevice().SafeCastTo<I2cDeviceProperty>();

    // パッドオブジェクトとセッションの紐づけ解除
    nn::ddsf::CloseSession(this);

    // 最後のセッションだった場合は FinalizeDevice を呼ぶ
    if ( !device.HasAnyOpenSession() )
    {
        auto& driver = device.GetDriver().SafeCastTo<II2cDriver>();
        driver.FinalizeDevice(&device);
    }
}

nn::Result I2cSessionContext::Send(const void* pData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT
{
    auto& device = GetDevice().SafeCastTo<I2cDeviceProperty>();
    std::lock_guard<nn::os::SdkMutex> lock(device.GetDriver().SafeCastTo<II2cDriver>().GetTransactionOrderMutex());

    return ExecuteTransactionWithRetry(nullptr, Command::Send, pData, dataBytes, inOption);
}

nn::Result I2cSessionContext::Receive(void* pOutData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT
{
    auto& device = GetDevice().SafeCastTo<I2cDeviceProperty>();
    std::lock_guard<nn::os::SdkMutex> lock(device.GetDriver().SafeCastTo<II2cDriver>().GetTransactionOrderMutex());

    return ExecuteTransactionWithRetry(pOutData, Command::Receive, nullptr, dataBytes, inOption);
}

nn::Result I2cSessionContext::ExecuteCommandList(void* pOutReceiveBuffer, size_t receiveBufferSize, const void* pCommandList, size_t commandListLength) NN_NOEXCEPT
{
    auto& device = GetDevice().SafeCastTo<I2cDeviceProperty>();
    std::lock_guard<nn::os::SdkMutex> lock(device.GetDriver().SafeCastTo<II2cDriver>().GetTransactionOrderMutex());

     // 未実行コマンドの先頭アドレス
    const uint8_t* pCurrentCommand = static_cast<const uint8_t*>(pCommandList);

    // コマンドリストの終端
    const uint8_t* const pCommandLast = pCurrentCommand + commandListLength;

    // 受信前バッファの書き込み先先頭アドレス
    uint8_t* pCurrentReceiveBuffer = static_cast<uint8_t*>(pOutReceiveBuffer);

    while ( pCurrentCommand < pCommandLast )
    {
        // byte1 を見てコマンドを判定
        const uint8_t Cmd = (*pCurrentCommand & nn::i2c::detail::CommonCommandFormat::CommandHeader::Mask) >> nn::i2c::detail::CommonCommandFormat::CommandHeader::Pos;

        NN_ABORT_UNLESS(Cmd == nn::i2c::detail::CommandId_Send
            || Cmd == nn::i2c::detail::CommandId_Receive
            || Cmd == nn::i2c::detail::CommandId_Extension);

        // コマンド別に byte1 以降のコマンドリストをパースして実行
        switch ( Cmd )
        {
        case nn::i2c::detail::CommandId_Send:
            NN_RESULT_DO(SendHandler(&pCurrentCommand, &pCurrentReceiveBuffer));
            break;
        case nn::i2c::detail::CommandId_Receive:
            NN_RESULT_DO(ReceiveHandler(&pCurrentCommand, &pCurrentReceiveBuffer));
            break;
        case nn::i2c::detail::CommandId_Extension:
            NN_RESULT_DO(ExtensionHandler(&pCurrentCommand, &pCurrentReceiveBuffer));
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result I2cSessionContext::SendHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer) NN_NOEXCEPT
{
    // byte1
    const nn::i2c::TransactionOption Option = static_cast<nn::i2c::TransactionOption>(
        ((**pCurrentCmd & nn::i2c::detail::SendCommandFormat::StartCondition::Mask) ? nn::i2c::TransactionOption_StartCondition : 0) |
        ((**pCurrentCmd & nn::i2c::detail::SendCommandFormat::StopCondition::Mask) ? nn::i2c::TransactionOption_StopCondition : 0));

    (*pCurrentCmd)++;

    // byte2
    const size_t DataBytes = (**pCurrentCmd & nn::i2c::detail::SendCommandFormat::DataBytes::Mask) >> nn::i2c::detail::SendCommandFormat::DataBytes::Pos;

    (*pCurrentCmd)++;

    // byte3 -
    // コマンド発行
    NN_RESULT_DO(ExecuteTransactionWithRetry(nullptr, Command::Send, *pCurrentCmd, DataBytes, Option));

    (*pCurrentCmd) += DataBytes;

    NN_RESULT_SUCCESS;
}

nn::Result I2cSessionContext::ReceiveHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer) NN_NOEXCEPT
{
    // byte1
    const nn::i2c::TransactionOption Option = static_cast<nn::i2c::TransactionOption>(
        ((**pCurrentCmd & nn::i2c::detail::ReceiveCommandFormat::StartCondition::Mask) ? nn::i2c::TransactionOption_StartCondition : 0) |
        ((**pCurrentCmd & nn::i2c::detail::ReceiveCommandFormat::StopCondition::Mask) ? nn::i2c::TransactionOption_StopCondition : 0));

    (*pCurrentCmd)++;

    // byte2
    const size_t DataBytes = (**pCurrentCmd & nn::i2c::detail::ReceiveCommandFormat::DataBytes::Mask) >> nn::i2c::detail::ReceiveCommandFormat::DataBytes::Pos;

    (*pCurrentCmd)++;

    // コマンド発行
    NN_RESULT_DO(ExecuteTransactionWithRetry(*pCurrentReceiveBuffer, Command::Receive, nullptr, DataBytes, Option));

    (*pCurrentReceiveBuffer) += DataBytes;

    NN_RESULT_SUCCESS;
}

nn::Result I2cSessionContext::ExtensionHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer) NN_NOEXCEPT
{
    // byte1
    const uint8_t SubCommand = (**pCurrentCmd & nn::i2c::detail::CommonCommandFormat::SubCommandHeader::Mask) >> nn::i2c::detail::CommonCommandFormat::SubCommandHeader::Pos;

    (*pCurrentCmd)++;

    switch ( SubCommand )
    {
    case nn::i2c::detail::SubCommandId_Sleep:
        {
            // byte2
            const size_t DataBytes = (**pCurrentCmd & nn::i2c::detail::SleepCommandFormat::SleepMicroSeconds::Mask) >> nn::i2c::detail::SleepCommandFormat::SleepMicroSeconds::Pos;

            (*pCurrentCmd)++;

            // コマンド発行
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(DataBytes));
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result I2cSessionContext::ExecuteTransactionWithRetry(void* pOutData, Command command, const void* pInData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT
{
    nn::Result result;
    auto& device = GetDevice().SafeCastTo<I2cDeviceProperty>();

    int retryCount = 0;

    for ( ; ; )
    {
        switch ( command )
        {
        case Command::Send:
            NN_SDK_ASSERT_NOT_NULL(pInData);
            result = device.GetDriver().SafeCastTo<II2cDriver>().Send(&device, pInData, dataBytes, inOption);
            break;
        case Command::Receive:
            NN_SDK_ASSERT_NOT_NULL(pOutData);
            result = device.GetDriver().SafeCastTo<II2cDriver>().Receive(pOutData, &device, dataBytes, inOption);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultTimeout)
        {
            ++retryCount;
            if ( retryCount <= m_MaxRetryCount )
            {
                NN_DETAIL_I2C_ERROR("timeout occur retry (%d)\n", retryCount);
                nn::os::SleepThread(m_RetryInterval);
                continue;
            }
            NN_RESULT_THROW(nn::i2c::ResultBusBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }

    return result;
}

nn::Result I2cSessionContext::SetRetryPolicy(int maxRetryCount, int retryIntervalMicroSeconds) NN_NOEXCEPT
{
    m_MaxRetryCount = maxRetryCount;
    m_RetryInterval = nn::TimeSpan::FromMicroSeconds(retryIntervalMicroSeconds);
    NN_RESULT_SUCCESS;
}

}}}} // nn::i2c::driver::detail
