﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>

#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>
#include <nn/nn_SdkAssert.h>

#include <nn/devicecode/devicecode_Predefined.i2c.h>
#include <nn/ddsf/ddsf_CastSafe.h>
#include <nn/ddsf/ddsf_DeviceCodeEntryManager.h>
#include <nn/i2c/driver/detail/dt/i2c_DeviceCodeNodeParser.h>
#include <nn/i2c/driver/detail/i2c_I2cSessionContext.h>
#include <nn/i2c/driver/i2c_I2cDeviceProperty.h>
#include <nn/i2c/driver/i2c_II2cDriver.h>
#include <nn/result/result_HandlingUtility.h>

#include "i2c_Core.h"

NN_DDSF_CAST_SAFE_DEFINE(nn::i2c::driver::I2cDeviceProperty, nn::ddsf::IDevice);
NN_DDSF_CAST_SAFE_DEFINE(nn::i2c::driver::II2cDriver, nn::ddsf::IDriver);
NN_DDSF_CAST_SAFE_DEFINE(nn::i2c::driver::detail::I2cSessionContext, nn::ddsf::ISession);

namespace {

nn::i2c::driver::II2cDriver::List g_I2cDriverList;
nn::i2c::driver::detail::dt::DeviceCodeNodeRootParser g_I2cDeviceCodeNodeRootParser("subsys-i2c");
std::array<nn::ddsf::DeviceCodeEntryHolder, 64> g_I2cDeviceCodeEntryBuffer;
nn::ddsf::DeviceCodeEntryManager g_I2cDeviceCodeEntryManager(&g_I2cDeviceCodeEntryBuffer);

enum class DriverInitializationState
{
    NotInitialized,
    Initializing,
    Initialized
};

DriverInitializationState g_DriverInitializationState = DriverInitializationState::NotInitialized;

}

namespace nn { namespace i2c { namespace driver { namespace detail {

void RegisterDriver(II2cDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDriver);
    NN_SDK_ASSERT(g_DriverInitializationState == DriverInitializationState::NotInitialized
        || g_DriverInitializationState == DriverInitializationState::Initializing);
    g_I2cDriverList.push_back(*pDriver);
}

void InitializeDrivers() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(g_DriverInitializationState, DriverInitializationState::NotInitialized);

    g_DriverInitializationState = DriverInitializationState::Initializing;

    // デバイスコードノードのパース実施
    g_I2cDeviceCodeNodeRootParser.ParseAll();

    for ( auto&& driver : g_I2cDriverList )
    {
        driver.SafeCastTo<II2cDriver>().InitializeDriver();
    }

    g_DriverInitializationState = DriverInitializationState::Initialized;
}

nn::Result RegisterDeviceCodeNodeParser(nn::i2c::driver::detail::dt::IDeviceCodeNodeParser* pParser) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pParser);
    NN_SDK_ASSERT(g_DriverInitializationState != DriverInitializationState::Initialized,
        "Device code node parser MUST be registered before nn::i2c::driver::Initialize() is called or during the driver initialization function II2cDriver::InitializeDriver()\n");
    g_I2cDeviceCodeNodeRootParser.RegisterParser(pParser);
    NN_RESULT_SUCCESS;
}

nn::Result RegisterDeviceCode(nn::DeviceCode deviceCode, I2cDeviceProperty* pDevice) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);
    NN_SDK_ASSERT(g_DriverInitializationState == DriverInitializationState::Initializing,
        "Device code entry MUST be registered during the driver initialization function II2cDriver::InitializeDriver()\n");
    NN_RESULT_DO(g_I2cDeviceCodeEntryManager.Add(deviceCode, pDevice));
    NN_RESULT_SUCCESS;
}

II2cDriver::List& GetDriverList() NN_NOEXCEPT
{
    return g_I2cDriverList;
}

nn::Result FindDevice(I2cDeviceProperty** ppOutDevice, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_DriverInitializationState == DriverInitializationState::Initialized,
        "FindDevice MUST be called after nn::i2c::driver::Initialize() is done\n");
    NN_SDK_ASSERT_NOT_NULL(ppOutDevice);
    nn::ddsf::IDevice* pDevice;
    NN_RESULT_DO(g_I2cDeviceCodeEntryManager.FindDevice(&pDevice, deviceCode));
    *ppOutDevice = pDevice->SafeCastToPointer<I2cDeviceProperty>();
    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
// ForDev 系の bus index, address 直指定 API 向けデバイスコード変換ユーティリティ
nn::Result FindDeviceByBusIndexAndAddress(I2cDeviceProperty** ppOutDevice, int busIndex, Bit16 address) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_DriverInitializationState == DriverInitializationState::Initialized,
        "FindDeviceByBusIndexAndAddress MUST be called after nn::i2c::driver::Initialize() is done\n");
    NN_SDK_ASSERT_NOT_NULL(ppOutDevice);
    if ( !ppOutDevice )
    {
        return nn::i2c::ResultUnknown();
    }

    nn::DeviceCode deviceCode(0);
    switch ( busIndex )
    {
    case 0:
        deviceCode = NN_DEVICECODE_I2C_BUS_1;
        break;
    case 1:
        deviceCode = NN_DEVICECODE_I2C_BUS_2;
        break;
    case 2:
        deviceCode = NN_DEVICECODE_I2C_BUS_3;
        break;
    case 3:
        deviceCode = NN_DEVICECODE_I2C_BUS_4;
        break;
    case 4:
        deviceCode = NN_DEVICECODE_I2C_BUS_5;
        break;
    case 5:
        deviceCode = NN_DEVICECODE_I2C_BUS_6;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    *ppOutDevice = nullptr;
    g_I2cDeviceCodeEntryManager.ForEachEntry(
        [&](nn::ddsf::DeviceCodeEntry* pEntry) NN_NOEXCEPT -> bool
        {
            NN_SDK_ASSERT_NOT_NULL(pEntry);
            auto& device = pEntry->GetDevice().SafeCastTo<I2cDeviceProperty>();
            if ( (device.GetDriver().SafeCastTo<II2cDriver>().GetDeviceCode() == deviceCode) && (device.GetAddress() == address) )
            {
                *ppOutDevice = &device;
                return false; // Break ForEach loop
            }
            return true;
        }
    );
    if ( !*ppOutDevice )
    {
        NN_DETAIL_I2C_ERROR("Unsupported device (bus index:0x%x , address:0x%x) is specified\n", busIndex, address);
        return nn::i2c::ResultDeviceNotFound();
    }
    NN_RESULT_SUCCESS;
}

}}}} // nn::i2c::driver::detail
