﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include <nn/i2c/driver/i2c.h>

#include "i2c_TargetSpec.h"

namespace {

const int InvalidIndex = -1;

// DeviceSettings をサーチして device と対応する index を返す関数
int GetIndex(nn::i2c::I2cDevice device) NN_NOEXCEPT
{
    for( int i = 0; i < nn::i2c::driver::detail::DeviceSettingsLength; i++ )
    {
        if( device == nn::i2c::driver::detail::DeviceSettings[i].device )
        {
            return i;
        }
    }
    return InvalidIndex;
}

}

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

bool IsSupported( I2cDevice device ) NN_NOEXCEPT
{
    return GetIndex(device) != InvalidIndex;
}

int GetBusIdx( I2cDevice device ) NN_NOEXCEPT
{
    int index = GetIndex(device);
    NN_ABORT_UNLESS( index != InvalidIndex, "The device isn't supported. Please call ForDev APIs instead.\n");
    return DeviceSettings[index].busIdx;
}

nn::Bit16 GetSlaveAddress( I2cDevice device ) NN_NOEXCEPT
{
    int index = GetIndex(device);
    NN_ABORT_UNLESS( index != InvalidIndex, "The device isn't supported. Please call ForDev APIs instead.\n");
    return DeviceSettings[index].slaveAddress;
}

AddressingMode GetAddressingMode( I2cDevice device ) NN_NOEXCEPT
{
    int index = GetIndex(device);
    NN_ABORT_UNLESS( index != InvalidIndex, "The device isn't supported. Please call ForDev APIs instead.\n");
    return DeviceSettings[index].addressingMode;
}

SpeedMode GetSpeedMode( I2cDevice device ) NN_NOEXCEPT
{
    int index = GetIndex(device);
    NN_ABORT_UNLESS( index != InvalidIndex, "The device isn't supported. Please call ForDev APIs instead.\n");
    return DeviceSettings[index].speedMode;
}

int GetRetryConunt(I2cDevice device) NN_NOEXCEPT
{
    int index = GetIndex(device);
    NN_ABORT_UNLESS(index != InvalidIndex, "The device isn't supported. Please call ForDev APIs instead.\n");
    return DeviceSettings[index].retryCount;
}

nn::TimeSpan GetRetryInterval(I2cDevice device) NN_NOEXCEPT
{
    int index = GetIndex(device);
    NN_ABORT_UNLESS(index != InvalidIndex, "The device isn't supported. Please call ForDev APIs instead.\n");
    return DeviceSettings[index].retryInterval;
}

} // detail
} // driver
} // i2c
} // nn
