﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_InterruptEvent.h>

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

// ハンドルする割り込み番号の数
const int InterruptCountMax = 6;

// 割り込み番号テーブル
const nn::os::InterruptName InterruptNameI2c1  = 32 + 38; // TRM では Name が I2C となっている(1 は付いてないので注意)
const nn::os::InterruptName InterruptNameI2c2  = 32 + 84;
const nn::os::InterruptName InterruptNameI2c3  = 32 + 92;
const nn::os::InterruptName InterruptNameI2c4  = 32 + 120;
const nn::os::InterruptName InterruptNameI2c5  = 32 + 53;
const nn::os::InterruptName InterruptNameI2c6  = 32 + 63;
const nn::os::InterruptName InterruptNameTable[InterruptCountMax] =
{
    InterruptNameI2c1,   // I2C1
    InterruptNameI2c2,   // I2C2
    InterruptNameI2c3,   // I2C3
    InterruptNameI2c4,   // I2C4
    InterruptNameI2c5,   // I2C5
    InterruptNameI2c6,   // I2C6
};

const int PowerBusIdx = 4;

// デバイス設定情報を格納した構造体
const DeviceSetting DeviceSettings[] =
{
    // Debug pad
    {
        I2cDevice::I2cDevice_ClassicController,  // Device name
        0,                                       // I2C1 (<=> Bus# -1 )
        0x52,                                    // 7'h52 (7 bit 表記)
        AddressingMode_BitWidth7,                // Addressing mode
        SpeedMode_Standard,                      // Speed mode
        0,                                       // Retry count (ドライバ側で対応されていて、リトライなしでよいなら 0)
        0                                        // Retry Interval
    },
    // Touch panel
    {
        I2cDevice::I2cDevice_Ftm3bd56,
        2,
        0x49,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        0,
        0
    },
    // Thermal Sensor
    {
        I2cDevice::I2cDevice_Tmp451,
        0,
        0x4c,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        0,
        0
    },
    // Thermal Sensor
    {
        I2cDevice::I2cDevice_Nct72,
        0,
        0x4c,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        0,
        0
    },
    // Audio codec
    {
        I2cDevice::I2cDevice_Alc5639,
        0,
        0x1c,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        0,
        0
    },
    // Power management unit - RTC
    {
        I2cDevice::I2cDevice_Max77620Rtc,
        4,
        0x68,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power management unit - Pmic
    {
        I2cDevice::I2cDevice_Max77620Pmic,
        4,
        0x3c,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power management unit - CPU
    {
        I2cDevice::I2cDevice_Max77621Cpu,
        4,
        0x1b,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power management unit - GPU
    {
        I2cDevice::I2cDevice_Max77621Gpu,
        4,
        0x1c,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Charger
    {
        I2cDevice::I2cDevice_Bq24193,
        0,
        0x6b,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Fuel gauge
    {
        I2cDevice::I2cDevice_Max17050,
        0,
        0x36,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power delivery controller
    {
        I2cDevice::I2cDevice_Bm92t30mwv,
        0,
        0x18,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor V_VDD15V0-HB
    {
        I2cDevice::I2cDevice_Ina226Vdd15v0Hb,
        1,
        0x40,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor CPU
    {
        I2cDevice::I2cDevice_Ina226VsysCpuDs,
        1,
        0x41,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor GPU
    {
        I2cDevice::I2cDevice_Ina226VsysGpuDs,
        1,
        0x44,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor LPDDR4
    {
        I2cDevice::I2cDevice_Ina226VsysDdrDs,
        1,
        0x45,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor V_SYS_AP
    {
        I2cDevice::I2cDevice_Ina226VsysAp,
        1,
        0x46,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor BackLight
    {
        I2cDevice::I2cDevice_Ina226VsysBlDs,
        1,
        0x47,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Ambient light sensor
    {
        I2cDevice::I2cDevice_Bh1730,
        1,
        0x29,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor - Core
    {
        I2cDevice::I2cDevice_Ina226VsysCore,
        1,
        0x48,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor - SoC 1V8
    {
        I2cDevice::I2cDevice_Ina226Soc1V8,
        1,
        0x49,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor - LPDDR4 1V8
    {
        I2cDevice::I2cDevice_Ina226Lpddr1V8,
        1,
        0x4A,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor - REG1V32
    {
        I2cDevice::I2cDevice_Ina226Reg1V32,
        1,
        0x4B,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // Power monitor - VDD3V3-SYS
    {
        I2cDevice::I2cDevice_Ina226Vdd3V3Sys,
        1,
        0x4D,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // HDMI DDC
    {
        I2cDevice::I2cDevice_HdmiDdc,
        3,
        0x50,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        0,
        0
    },
    // HDMI SCDC
    {
        I2cDevice::I2cDevice_HdmiScdc,
        3,
        0x54,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        0,
        0
    },
    // HDMI HDCP
    {
        I2cDevice::I2cDevice_HdmiHdcp,
        3,
        0x3A,
        AddressingMode_BitWidth7,
        SpeedMode_Standard,
        0,
        0
    },
    // Fan53528
    {
        I2cDevice::I2cDevice_Fan53528,
        4,
        0xA4,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        0,
        0
    },
    // Max77812 (3phase)
    {
        I2cDevice::I2cDevice_Max77812_3_1,
        4,
        0x31,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        0,
        0
    },
    // Max77812 (2phase)
    {
        I2cDevice::I2cDevice_Max77812_2_1_1,
        4,
        0x33,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        0,
        0
    },
    // Power monitor V_VDD-DDR-0V6
    {
        I2cDevice::I2cDevice_Ina226VddDdr0V6,
        1,
        0x4e,
        AddressingMode_BitWidth7,
        SpeedMode_Fast,
        3,
        nn::TimeSpan::FromMilliSeconds(5)
    },
    // ここにデバイスを追記してください。
    // pcv のポートに対応するものを追加する場合、i2c_ServiceType.h の変更も必要です
};


const int DeviceSettingsLength = (sizeof(DeviceSettings) / sizeof(DeviceSettings[0]));

const int MaxBuses             = 6;         // [Ref1] 31.0 物理的なバス数は6つ

} // detail
} // driver
} // i2c
} // nn
