﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <mutex>

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/pcv/pcv.h>

#include <nn/os/os_ThreadApi.h>

#include <nn/i2c/driver/i2c.h>
#include <nn/i2c/driver/i2c_BusDev.h>
#include <nn/i2c/detail/i2c_Log.h>

#include "i2c_Util.h"
#include "i2c_TargetSpec.h"
#include "i2c_ResourceManager.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

void ResourceManager::Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_InitializeCountMutex );

    NN_SDK_ASSERT( m_InitializeCount < INT_MAX ); // 値域チェック

    m_InitializeCount++;

    // 初期化済みならば何も行わない
    if ( m_InitializeCount > 1)
    {
        NN_DETAIL_I2C_LOG("%s(%d) Already Initialized. COUNT=%d\n", __FUNCTION__, __LINE__ , m_InitializeCount );
        return;
    }
    NN_SDK_REQUIRES( m_InitializeCount == 1); // 念のためチェック

    NN_DETAIL_I2C_LOG("%s(%d) Initialize Done.\n", __FUNCTION__, __LINE__ );
}

void ResourceManager::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_InitializeCountMutex );

    NN_SDK_ASSERT( m_InitializeCount > INT_MIN ); // 値域チェック

    m_InitializeCount--;

    // 終了回数が初期化された回数に達していない場合は何も行わない
    if ( m_InitializeCount > 0 )
    {
        NN_DETAIL_I2C_LOG("%s(%d) Not Finalized. COUNT=%d.\n", __FUNCTION__, __LINE__ , m_InitializeCount );
        return;
    }
    NN_SDK_REQUIRES( m_InitializeCount == 0);

    std::lock_guard<nn::os::Mutex> lockSession( m_SessionIsOpenMutex );

    for( int i = 0; i < MaxDriverSessions; i++ )
    {
        m_Sessions[i].Close();
    }

    NN_DETAIL_I2C_LOG("%s(%d) Finalize Done.\n", __FUNCTION__, __LINE__ );
}

int ResourceManager::FindFreeSessionId() const NN_NOEXCEPT
{
    for(int i = 0; i < MaxDriverSessions; i++ )
    {
        if (!m_Sessions[i].IsOpened())
        {
            return i; // 見つけた
        }
    }

    return -1; // 全部オープン済だった
}

void ResourceManager::DumpSessions() NN_NOEXCEPT
{
    for(int i = 0; i < MaxDriverSessions; i++ )
    {
        m_Sessions[i].Dump();
    }
}

void ResourceManager::OpenSession(I2cSession* pOutSession, int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode,
                                  int retryCount, nn::TimeSpan retryInterval) NN_NOEXCEPT
{
    m_SessionIsOpenMutex.Lock();

    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_REQUIRES(busIdx < MaxBuses);

    int sessionId = FindFreeSessionId();
    bool validSessionId = (0 <= sessionId && sessionId < MaxDriverSessions);
// NX 専用処理
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // NX では I2C2 と I2C3 を始めて使う場合には LDO6 の電源を入れる必要がある。
    // これは全バスに対するロック中にチェックする
    bool IsNeededLdo6PowerCycleOn = false;

    if( (busIdx == 1 || busIdx == 2) && (m_BusAccessor[1].GetUserCount() == 0 && m_BusAccessor[2].GetUserCount() == 0))
    {
        IsNeededLdo6PowerCycleOn = true;
    }
#endif

    if ( !validSessionId )
    {
        DumpSessions();
    }

    NN_ABORT_UNLESS(validSessionId, "No available sessions on I2C library");

    pOutSession->_sessionId = sessionId;
    pOutSession->_busIdx = busIdx;

    m_Sessions[sessionId].Open(busIdx, slaveAddress, addressingMode, speedMode, &m_BusAccessor[busIdx], retryCount, retryInterval);

    NN_DETAIL_I2C_LOG("%s(%d) OpenSession at sessionId=%d\n", __FUNCTION__, __LINE__, sessionId );

    // ここで UnLock しておかないと、次の Pcv の処理及び Session::Start() 内部で pcv を呼ぶ処理がロックを取り合ってしまう。
    m_SessionIsOpenMutex.Unlock();

    m_Sessions[sessionId].Start();

// NX 専用処理
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    if(IsNeededLdo6PowerCycleOn)
    {
        nn::pcv::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageValue(nn::pcv::PowerDomain_Max77620_Ldo6, 2900000));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo6, true));
        nn::pcv::Finalize();

        // LDO6 が立ち上がるまでには 0.56 ms の待ちが必要
        nn::os::SleepThread(::nn::TimeSpan::FromMicroSeconds(560));
        NN_DETAIL_I2C_INFO("Set LDO6 voltage to %d\n",2900000);
    }
#endif
}


void ResourceManager::CloseSession( const I2cSession& session ) NN_NOEXCEPT
{
    m_SessionIsOpenMutex.Lock();

    auto      sessionId = session._sessionId; // ユーザー領域へのアクセスは最小限に

    NN_SDK_ASSERT( m_Sessions[sessionId].IsOpened(), "Session Already Closed");

    m_Sessions[sessionId].Close();

    NN_DETAIL_I2C_LOG("%s(%d) CloseSession at sessionId=%d\n", __FUNCTION__, __LINE__, sessionId );
// NX 専用処理
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // I2C2 と I2C3 の使用者がいなくなったら LDO6 の電源を落とす
    // これは全バスに対するロック中にチェックする
    bool IsNeededLdo6PowerCycleOff = false;

    if( (session._busIdx == 1 || session._busIdx == 2) && (m_BusAccessor[1].GetUserCount() == 0 && m_BusAccessor[2].GetUserCount() == 0))
    {
        IsNeededLdo6PowerCycleOff = true;
    }
#endif

    // ここで UnLock しておかないと、次の Pcv の処理がロックを取り合ってしまう。
    m_SessionIsOpenMutex.Unlock();

// NX 専用処理
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    if(IsNeededLdo6PowerCycleOff)
    {
        nn::pcv::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo6, false));
        nn::pcv::Finalize();
    }
#endif
}

StaticMutex* ResourceManager::GetTransactionOrderMutex( int busIdx ) NN_NOEXCEPT
{
    return &m_TransactionOrderMutex[busIdx];
}

void ResourceManager::SuspendBuses() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_InitializeCount > 0);

    if (!m_IsSuspended)
    {
        {
            std::lock_guard<nn::os::Mutex> lock(m_SessionIsOpenMutex);
            //NN_DETAIL_I2C_INFO("Suspending\n");
            m_IsSuspended = true;
            for (int i = 0; i < MaxBuses; i++)
            {
                if (i != PowerBusIdx && m_BusAccessor[i].GetUserCount() > 0)
                {
                    m_BusAccessor[i].Suspend();
                }
            }
        }
        // 以降の処理はロックを解除する必要がある

        // NX 専用処理
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        // 最後に LDO6 の電源を落とす
        NN_DETAIL_I2C_INFO("Disable LDO6 Voltage\n");
        nn::pcv::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo6, false));
        nn::pcv::Finalize();
#endif
    }
}

void ResourceManager::SuspendPowerBuses() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_SessionIsOpenMutex);
    NN_SDK_ASSERT(m_InitializeCount > 0);

    if (!m_IsPowerBusSuspended)
    {
        //NN_DETAIL_I2C_INFO("PowerBus Suspending\n");
        m_IsPowerBusSuspended = true;
        for (int i = 0; i < MaxBuses; i++)
        {
            if (i == PowerBusIdx && m_BusAccessor[i].GetUserCount() > 0)
            {
                m_BusAccessor[i].Suspend();
            }
        }
    }
}

void ResourceManager::ResumeBuses() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_InitializeCount > 0);

    if (m_IsSuspended)
    {
         //NN_DETAIL_I2C_INFO("Resuming\n");

        // NX 専用処理
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        // 最初に LDO6 の電源を入れる
        if (m_BusAccessor[1].GetUserCount() > 0 || m_BusAccessor[2].GetUserCount() > 0)
        {
            nn::pcv::Initialize();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageValue(nn::pcv::PowerDomain_Max77620_Ldo6, 2900000));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo6, true));
            nn::pcv::Finalize();

            // LDO6 が立ち上がるまでには 0.56 ms (2.9V) +  1 ms (1.8V) の待ちが必要
            nn::os::SleepThread(::nn::TimeSpan::FromMicroSeconds(560));
            nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(1));
            NN_DETAIL_I2C_INFO("Set LDO6 voltage to %d\n", 2900000);
        }
#endif
        std::lock_guard<nn::os::Mutex> lock(m_SessionIsOpenMutex);

        for (int i = 0; i < MaxBuses; i++)
        {
            if (i != PowerBusIdx && m_BusAccessor[i].GetUserCount() > 0)
            {
                m_BusAccessor[i].Resume();
            }
        }
        m_IsSuspended = false;
    }
}

void ResourceManager::ResumePowerBuses() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_SessionIsOpenMutex);

    NN_SDK_ASSERT(m_InitializeCount > 0);

    if (m_IsPowerBusSuspended)
    {
        //NN_DETAIL_I2C_INFO("PowerBus Resuming\n");
        for (int i = 0; i < MaxBuses; i++)
        {
            if (i == PowerBusIdx && m_BusAccessor[i].GetUserCount() > 0)
            {
                m_BusAccessor[i].Resume();
            }
        }
        m_IsPowerBusSuspended = false;
    }
}

} // detail
} // driver
} // i2c
} // nn
