﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file Pinmux 関係のレジスタ定義を切り出したファイル
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/dd.h>
#include "i2c_DdUtil.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

struct PinmuxRegInfo
{
public:
    void Set(int busIdx) NN_NOEXCEPT
    {
        /**
         * @brief Pinmux レジスタが含まれる APB_MISC の物理アドレス ([Ref1] 2.0)
         */
        static const nn::dd::PhysicalAddress   ApbMiscPhysicalAddress = 0x070000000ull;
        static const size_t                    ApbMiscAddressSize     = 0x4000;

        // Pinmux register <=> PINMUX_*_I2C_SCL ([Ref1] 8.12)
        static const uintptr_t SdlRegOffsetTable[] =
        {
            0x31a4, // I2C1
            0x3250, // I2C2
            0x3290, // I2C3
            0x3114, // I2C4
            0x0000, // I2C5 Pinmux の定義が無いため設定しない
            0x0000  // I2C6 Pinmux の定義が無いため設定しない
        };
        static const uint32_t SdlRegValueTable[] =
        {
            // OD       E_INPUT    TRISTATE   PUPD        PM
            (1 <<  6) | (1 << 5) | (0 << 4) | (0 << 2) | (0 << 0), // I2C1
            (1 <<  6) | (1 << 5) | (0 << 4) | (0 << 2) | (0 << 0), // I2C2
            (1 <<  6) | (1 << 5) | (0 << 4) | (0 << 2) | (1 << 0), // I2C3
                        (1 << 5) | (0 << 4) | (0 << 2) | (0 << 0), // I2C4 OD ビットの定義がない
                                                         (0 << 0), // I2C5 Pinmux の定義が無いため設定しない
                                                         (0 << 0)  // I2C6 Pinmux の定義が無いため設定しない
        };
        static const uint32_t SdlRegMaskTable[] =
        {
            0x7f, // I2C1
            0x7f, // I2C2
            0x7f, // I2C3
            0x3f, // I2C4
               0, // I2C5 Pinmux の定義が無いため設定しない (Mask が 0 のときは設定処理はスキップされる)
               0  // I2C6 Pinmux の定義が無いため設定しない (Mask が 0 のときは設定処理はスキップされる)
        };
        NN_UNUSED(SdlRegMaskTable);

        // Pinmux register <=> PINMUX_*_I2C_SDA ([Ref1] 8.12)
        static const uintptr_t SdaRegOffsetTable[] =
        {
            0x31a0, // I2C1
            0x3254, // I2C2
            0x3294, // I2C3
            0x3118, // I2C4
            0x0000, // I2C5 Pinmux の定義が無いため設定しない
            0x0000  // I2C6 Pinmux の定義が無いため設定しない
        };
        static const uint32_t SdaRegValueTable[] =
        {
            // OD       E_INPUT    TRISTATE   PUPD        PM
            (1 <<  6) | (1 << 5) | (0 << 4) | (0 << 2) | (0 << 0), // I2C1
            (1 <<  6) | (1 << 5) | (0 << 4) | (0 << 2) | (0 << 0), // I2C2
            (1 <<  6) | (1 << 5) | (0 << 4) | (0 << 2) | (1 << 0), // I2C3
                        (1 << 5) | (0 << 4) | (0 << 2) | (0 << 0), // I2C4 OD ビットの定義がない
                                                         (0 << 0), // I2C5 Pinmux の定義が無いため設定しない
                                                         (0 << 0)  // I2C6 Pinmux の定義が無いため設定しない
        };
        static const uint32_t SdaRegMaskTable[] =
        {
            0x7f, // I2C1
            0x7f, // I2C2
            0x7f, // I2C3
            0x3f, // I2C4
               0, // I2C5 Pinmux の定義が無いため設定しない (Mask が 0 のときは設定処理はスキップされる)
               0  // I2C6 Pinmux の定義が無いため設定しない (Mask が 0 のときは設定処理はスキップされる)
        };
        NN_UNUSED(SdaRegMaskTable);

        NN_SDK_ASSERT(0 <= busIdx && busIdx < static_cast<int>( sizeof(SdlRegOffsetTable) / sizeof(SdlRegOffsetTable[0]) ));

        // IO マッピングを取得する
        uintptr_t baseAddr = GetVirtualAddress(ApbMiscPhysicalAddress, ApbMiscAddressSize); // APB_MISC

        pScl        = reinterpret_cast<volatile uint32_t*>(baseAddr + SdlRegOffsetTable[busIdx]);
        sclBitValue = SdlRegValueTable[busIdx];
        sclBitMask  = SdlRegValueTable[busIdx];
        pSda        = reinterpret_cast<volatile uint32_t*>(baseAddr + SdaRegOffsetTable[busIdx]);
        sdaBitValue = SdaRegValueTable[busIdx];
        sdaBitMask  = SdaRegValueTable[busIdx];
    }

public:
    volatile uint32_t*  pScl;
    uint32_t            sclBitValue;
    uint32_t            sclBitMask;
    volatile uint32_t*  pSda;
    uint32_t            sdaBitValue;
    uint32_t            sdaBitMask;
};

} // detail
} // driver
} // i2c
} // nn
