﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http/stream/http_WebApiAccessorBase.h>

#include <cctype>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Version.h>
#include <nn/http/http_Result.h>
#include <nn/http/detail/http_Log.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ssl/ssl_Api.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "http_StreamUtility.h"

namespace nn { namespace http { namespace stream {
namespace {
enum class HttpVersion
{
    Version1_0 = 0x0100,
    Version1_1 = 0x0101,
    Version2_0 = 0x0200,
};

const char* GetHttpHeaderLeadingFromVersion(size_t* pOutSize, HttpVersion version) NN_NOEXCEPT
{
    *pOutSize = sizeof("HTTP/1.0 ");
    switch (version)
    {
    case HttpVersion::Version1_0:
        return "HTTP/1.0 ";
    case HttpVersion::Version1_1:
        return "HTTP/1.1 ";
    case HttpVersion::Version2_0:
        return "HTTP/2.0 ";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

template <int Length>
int ExtractStatusCode(const char* ptr) NN_NOEXCEPT
{
    int code = 0;
    for (int i = 0; i < Length; ++ i)
    {
        auto c = ptr[i];
        if (!std::isdigit(c))
        {
            return -1;
        }
        code = (code * 10) + (c - '0');
    }
    return code;
}

#define NN_HTTP_SDK_VERSION_STRING(major, minor, micro, relstep) \
    # major "." # minor "." # micro "." # relstep

const char* GetUserAgent() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(os::SdkMutex, s_Lock);
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

#define NN_HTTP_UA_AGENT     "libcurl"
#define NN_HTTP_UA_MODULE    "nnHttp"

#if defined(NN_BUILD_CONFIG_SPEC_GENERIC)
#define NN_HTTP_UA_PLATFORM  "5797bc97-d32c-48ad-b47e-3847bda0db7c" // Generic Windows
#elif defined(NN_BUILD_CONFIG_SPEC_NX)
#if defined(NN_BUILD_CONFIG_OS_WIN)
#define NN_HTTP_UA_PLATFORM  "159b8d57-d7af-4ce6-823e-c38d1d661918" // NX Windows
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
#define NN_HTTP_UA_PLATFORM  "789f928b-138e-4b2f-afeb-1acae821d897" // NX Horizon
#else
#   error "Unsupported OS"
#endif // OS
#else
#   error "Unsupported Spec"
#endif // Spec

    typedef char UserAgentStringType[256];
    NN_FUNCTION_LOCAL_STATIC(UserAgentStringType, g_UserAgent, = {});

    NN_FUNCTION_LOCAL_STATIC(bool, g_Initialized, = false);
    if (!g_Initialized)
    {

#if defined(NN_BUILD_CONFIG_SPEC_NX)
        auto l = util::SNPrintf(
            g_UserAgent, sizeof(g_UserAgent),
            "%s (%s; %s; SDK %d.%d.%d.%d; Add-on %d.%d.%d.%d)",
            NN_HTTP_UA_AGENT, NN_HTTP_UA_MODULE, NN_HTTP_UA_PLATFORM,
            NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP,
            NN_NX_ADDON_VERSION_MAJOR, NN_NX_ADDON_VERSION_MINOR, NN_NX_ADDON_VERSION_MICRO, NN_NX_ADDON_VERSION_RELSTEP);
#else
        auto l = util::SNPrintf(
            g_UserAgent, sizeof(g_UserAgent),
            "%s (%s; %s; SDK %d.%d.%d.%d)",
            NN_HTTP_UA_AGENT, NN_HTTP_UA_MODULE, NN_HTTP_UA_PLATFORM,
            NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP);
#endif
        NN_ABORT_UNLESS(l < sizeof(g_UserAgent), "[nn::http] Error: Insufficient buffer for user agent: required %d bytes\n", l + 1);

        NN_DETAIL_HTTP_TRACE("[nn::http] User agent is initialized as \"%s\"\n", g_UserAgent);
        g_Initialized = true;
    }
    return g_UserAgent;
}
} // ~namespace nn::http::stream::<anonymous>

bool TryExtractStatusCodeFromHeader(long* pOut, const char* header, const size_t Length) NN_NOEXCEPT
{
    static const HttpVersion Versions[] = {
        HttpVersion::Version1_0,
        HttpVersion::Version1_1,
        HttpVersion::Version2_0,
    };

    bool found = false;
    size_t size = 0;
    for (auto& Version : Versions)
    {
        auto leading = GetHttpHeaderLeadingFromVersion(&size, Version);
        if (true
            && Length >= size
            && util::Strnicmp(header, leading, static_cast<int>(size - 1)) == 0)
        {
            found = true;
        }
    }
    if (!found)
    {
        // HTTP ステータスコードではない
        return false;
    }

    auto status = header + (size - 1);
    int statusCode;
    if (!(Length - (size - 1) > 3 && status[3] == ' ' && (statusCode = ExtractStatusCode<3>(status)) >= 0))
    {
        // 未知のステータスコードである
#if !defined(NN_SDK_BUILD_RELEASE)
        NN_DETAIL_HTTP_WARN("[http] Unknown HTTP status code: \"%*s\"\n", Length, header);
#endif
        statusCode = -1;
    }
    *pOut = statusCode;
    return true;
}

/* --------------------------------------------------------------------------------------------
    Callback
 */
size_t WebApiAccessorBase::HeaderFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT
{
    NN_UNUSED(ptr);
    NN_UNUSED(pContext);
    const auto InputBytes = unitBytes * count;
#if 0
    if (true
        && InputBytes > 0
        && ptr[0] != '\n'
        && ptr[0] != '\r'
        && ptr[0] != ' ')
    {
        NN_DETAIL_HTTP_TRACE("HEADER ------------------------\n");
        NN_DETAIL_HTTP_TRACE("%.*s\n", InputBytes, ptr);
    }
#endif

    auto& obj = *reinterpret_cast<WebApiAccessorBase*>(pContext);

    long statusCode;
    if (TryExtractStatusCodeFromHeader(&statusCode, ptr, InputBytes))
    {
        // 100 のケースもあるので最後のステータスコードを採用する
        obj.m_StatusCode = statusCode;
        obj.m_IsStatusCodeExtracted = true;
        if (!(obj.m_StatusCode > 0))
        {
            return CallbackErrorValue;
        }
    }
    return InputBytes;
}

size_t WebApiAccessorBase::WriteFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT
{
    NN_UNUSED(ptr);
    NN_UNUSED(pContext);
    const auto InputBytes = unitBytes * count;
#if 0
    if (true
        && InputBytes > 0
        && ptr[0] != '\n'
        && ptr[0] != '\r'
        && ptr[0] != ' ')
    {
        NN_DETAIL_HTTP_TRACE("BODY ------------------------\n");
        NN_DETAIL_HTTP_TRACE("%.*s\n", InputBytes, ptr);
    }
#endif
    return InputBytes;
}

int WebApiAccessorBase::XferInfoFunction(void* pContext, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow) NN_NOEXCEPT
{
    NN_UNUSED(dlTotal);
    NN_UNUSED(dlNow);
    NN_UNUSED(ulTotal);
    NN_UNUSED(ulNow);
    auto& obj = *reinterpret_cast<WebApiAccessorBase*>(pContext);
    if (obj.IsCanceled())
    {
        return 1; // Canceled
    }
    return 0;
}

CURLcode WebApiAccessorBase::DefaultSslCtxFunction(CURL* curlHandle, void* pSslContext, SslCtxFunctionContext* pFunctionContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFunctionContext);
    pFunctionContext->isCalled = true;
    if (!pFunctionContext->result.IsSuccess())
    {
        return CURLE_ABORTED_BY_CALLBACK;
    }

    auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
    auto r = context.Create(nn::ssl::Context::SslVersion_Auto);
    if (r.IsSuccess() && pFunctionContext->handler != nullptr)
    {
        // ユーザー指定のコンテキストハンドラを呼ぶ
        r = pFunctionContext->handler(curlHandle, context, pFunctionContext->userData);
    }

    if (!r.IsSuccess())
    {
        pFunctionContext->result = r;
        return CURLE_ABORTED_BY_CALLBACK;
    }
    return CURLE_OK;
}

/* --------------------------------------------------------------------------------------------
    実行の制御
 */
const WebApiAccessorBase::SslCtxFunctionContext WebApiAccessorBase::SslCtxFunctionContext::Default = {false, ResultSuccess(), nullptr, nullptr};

WebApiAccessorBase::WebApiAccessorBase(CURL* curlHandle, const util::Cancelable* pCancelable) NN_NOEXCEPT
    : util::CancelInjectable(pCancelable)
    , m_CurlHandle(curlHandle)
{
}
WebApiAccessorBase::~WebApiAccessorBase() NN_NOEXCEPT
{
    WebApiAccessorBase::Reset();
}

Result WebApiAccessorBase::Attach(CURLM* multi) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    auto mr = curl_multi_add_handle(multi, m_CurlHandle);
    NN_RESULT_DO(ConvertCurlMultiCodeToResult(mr));
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::Detach(CURLM* multi) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    auto mr = curl_multi_remove_handle(multi, m_CurlHandle);
    NN_RESULT_DO(ConvertCurlMultiCodeToResult(mr));
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::Pause(int bitmask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    return HandleCurlError(curl_easy_pause(m_CurlHandle, bitmask));
}

Result WebApiAccessorBase::GetResult() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    if (m_IsStatusCodeExtracted)
    {
        return HandleHttpStatus(m_StatusCode);
    }

    long httpCode = -1;
    NN_RESULT_DO(HandleCurlError(curl_easy_getinfo(m_CurlHandle, CURLINFO_RESPONSE_CODE, &httpCode)));
    return HandleHttpStatus(httpCode);
}
long WebApiAccessorBase::GetHttpCode() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    if (m_IsStatusCodeExtracted)
    {
        return m_StatusCode;
    }

    long httpCode;
    auto curle = curl_easy_getinfo(m_CurlHandle, CURLINFO_RESPONSE_CODE, &httpCode);
    return curle == CURLE_OK ? httpCode : -1;
}
Result WebApiAccessorBase::Perform() NN_NOEXCEPT
{
    return HandleCurlErrorWithContext(curl_easy_perform(m_CurlHandle));
}

/* --------------------------------------------------------------------------------------------
    エラー処理
 */
Result WebApiAccessorBase::GetSslResultFromHandle(CURLINFO curlInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT(curlInfo == CURLINFO_SSL_VERIFYRESULT || curlInfo == CURLINFO_SSL_HANDSHAKE_RESULT);
    long value;
    NN_RESULT_DO(HandleCurlError(curl_easy_getinfo(m_CurlHandle, curlInfo, &value)));
    Result r;
    NN_RESULT_DO(ssl::GetSslResultFromValue(&r, reinterpret_cast<char*>(&value), sizeof(value)));
    return r;
}
Result WebApiAccessorBase::HandleCurlError(CURLcode e) const NN_NOEXCEPT
{
    if (m_ErrorHandler.curlErrorHandler)
    {
        return m_ErrorHandler.curlErrorHandler(e);
    }
    return ConvertCurlCodeToResult(e);
}
Result WebApiAccessorBase::HandleHttpStatus(long status) const NN_NOEXCEPT
{
    if (m_ErrorHandler.httpStatusHandler)
    {
        return m_ErrorHandler.httpStatusHandler(status);
    }
    return ConvertHttpStatusToResult(status);
}
Result WebApiAccessorBase::HandleCurlErrorWithContext(CURLcode e) NN_NOEXCEPT
{
    switch (e)
    {
    case CURLE_SSL_CONNECT_ERROR:
        if (m_SslCtxFunctionContext.isCalled)
        {
            NN_RESULT_DO(GetSslResultFromHandle(CURLINFO_SSL_HANDSHAKE_RESULT));
        }
        break;
    case CURLE_PEER_FAILED_VERIFICATION:
        if (m_SslCtxFunctionContext.isCalled)
        {
            NN_RESULT_DO(GetSslResultFromHandle(CURLINFO_SSL_VERIFYRESULT));
        }
        break;
    case CURLE_ABORTED_BY_CALLBACK:
        if (IsCanceled())
        {
            NN_RESULT_THROW(ResultCanceled());
        }
        else if (m_SslCtxFunctionContext.isCalled)
        {
            NN_RESULT_DO(m_SslCtxFunctionContext.result);
        }
        break;
    default:
        break;
    }
    return HandleCurlError(e);
}

/* --------------------------------------------------------------------------------------------
    libcurl の設定
 */
#define NN_HTTP_SET_CURLOPT(handle, opt, ...) \
    do \
    { \
        CURLcode _r = curl_easy_setopt(handle, opt, __VA_ARGS__); \
        if (_r != CURLE_OK) \
        { \
            NN_DETAIL_HTTP_ERROR("[nn::http] curl_easy_setopt(" #opt ") failed (%d)\n", _r); \
            NN_RESULT_DO(HandleCurlError(_r)); \
        } \
    } while (NN_STATIC_CONDITION(false))


Result WebApiAccessorBase::SetDefault() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_Initialized);

    // Close でも reset されるので、連続してハンドルを使いまわすと怪しいことになりそうだが、そんなに重くないので大丈夫
    Reset();

    /* basic */
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_NOPROGRESS,        0L);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_VERBOSE,           0L);

    /* advance*/
    const long RedirectCountMax =  0L; // リダイレクトは 0 回まで
    const long LowSpeedLimit    =  1L; // 転送ビットレートは 1bps を下限とする
    const long LowSpeedTime     = 30L; // LowSpeedLimit が 30 秒つづくと通信を中断する
    const long ConnectTimeout   = 30L; // サーバとの接続(CONNECT)のタイムアウト (30秒)
    const long TimeoutSeconds   = 60L; // 通信全体でのタイムアウト (1分)
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_FOLLOWLOCATION,    1L);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_MAXREDIRS,         RedirectCountMax);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_LOW_SPEED_LIMIT,   LowSpeedLimit);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_LOW_SPEED_TIME,    LowSpeedTime);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_CONNECTTIMEOUT,    ConnectTimeout);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_TIMEOUT,           TimeoutSeconds);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYPEER,    0L); // server validation is disabled for testing
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYHOST,    0L); // host name validation is disabled for testing
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYPEER,    1L);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYHOST,    2L);
#else
#   error "Unsupported OS specified"
#endif
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYDATE,    0L); // 時刻検証無効

    /* callback */
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_HEADERFUNCTION,    HeaderFunction);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_HEADERDATA,        this);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_WRITEFUNCTION,     WriteFunction);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_WRITEDATA,         nullptr);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_XFERINFOFUNCTION,  XferInfoFunction);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_XFERINFODATA,      this);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_CTX_FUNCTION,  DefaultSslCtxFunction);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_CTX_DATA,      &m_SslCtxFunctionContext);

    /* user agent */
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_USERAGENT,         GetUserAgent());

    m_Initialized = true;
    NN_RESULT_SUCCESS;
}
void WebApiAccessorBase::Reset() NN_NOEXCEPT
{
    curl_easy_reset(m_CurlHandle);
    m_SslCtxFunctionContext = SslCtxFunctionContext::Default;
    m_ErrorHandler = {nullptr, nullptr};
    m_IsStatusCodeExtracted = false;
}
Result WebApiAccessorBase::SetErrorHandler(ErrorHandler handler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(handler.curlErrorHandler);
    NN_SDK_ASSERT_NOT_NULL(handler.httpStatusHandler);
    m_ErrorHandler = handler;
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetUserAgent(const char* ua) NN_NOEXCEPT
{
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_USERAGENT, ua);
    NN_RESULT_SUCCESS;
}

Result WebApiAccessorBase::SetDebugMode() NN_NOEXCEPT
{
#if !defined(NN_SDK_BUILD_RELEASE)
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_SKIP_DEFAULT_VERIFY, 1L);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_VERBOSE,                 1L);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYPEER,          0L);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_SSL_VERIFYHOST,          0L);
#endif
    NN_RESULT_SUCCESS;
}

Result WebApiAccessorBase::SetHeaderFunction(size_t(*headerFunction)(char*, size_t, size_t, void*), void* userData) NN_NOEXCEPT
{
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_HEADERFUNCTION,    headerFunction);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_HEADERDATA,        userData);
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetWriteFunction(size_t(*writeFunction)(char*, size_t, size_t, void*), void* userData) NN_NOEXCEPT
{
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_WRITEFUNCTION,     writeFunction);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_WRITEDATA,         userData);
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetReadFunction(size_t(*readFunction)(char*, size_t, size_t, void*), size_t size, void* userData) NN_NOEXCEPT
{
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_READFUNCTION,      readFunction);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_READDATA,          userData);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_UPLOAD, 1L);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_PUT, 1L);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_INFILESIZE_LARGE, static_cast<curl_off_t>(size));
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetSslContextHandler(SslContextHandler handler, void* userData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    m_SslCtxFunctionContext.handler = handler;
    m_SslCtxFunctionContext.userData = userData;
    NN_RESULT_SUCCESS;
}

Result WebApiAccessorBase::SetOpenSocketFunction(curl_socket_t(*openSocketFunction)(void*, curlsocktype, struct curl_sockaddr*), void* userData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_OPENSOCKETFUNCTION, openSocketFunction);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_OPENSOCKETDATA, userData);
    NN_RESULT_SUCCESS;
}

Result WebApiAccessorBase::SetUrl(const char* url) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_URL, url);
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetHeaders(const struct curl_slist* headers) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_HTTPHEADER, headers);
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetTimeoutSeconds(int32_t seconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_TIMEOUT, static_cast<long>(seconds));
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetHttpDelete() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_POSTFIELDS, nullptr);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_POSTFIELDSIZE, 0u);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_CUSTOMREQUEST, "DELETE");
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetHttpPatch(const char* patchData, bool copy) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_RESULT_DO(SetHttpPost(patchData, copy));
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_CUSTOMREQUEST, "PATCH");
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetHttpPost(const char* postData, bool copy) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, copy? CURLOPT_COPYPOSTFIELDS: CURLOPT_POSTFIELDS, postData);
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetHttpPost(const struct curl_httppost* pFormData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_HTTPPOST, pFormData);
    NN_RESULT_SUCCESS;
}
Result WebApiAccessorBase::SetHttpPut(const char* patchData, bool copy) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_RESULT_DO(SetHttpPost(patchData, copy));
    NN_HTTP_SET_CURLOPT(m_CurlHandle, CURLOPT_CUSTOMREQUEST, "PUT");
    NN_RESULT_SUCCESS;
}

#undef NN_HTTP_SET_CURLOPT

}}} // ~namespace nn::http::stream
