﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http.h>
#include <nn/nifm.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <curl/curl.h>

#include "http_Utility.h"

namespace nn { namespace http {

struct AllocateFunctions
{
    AllocateFunction   allocate;
    FreeFunction       free;
    ReAllocateFunction realloc;
} s_AllocateFunctions = { nullptr, nullptr, nullptr };

Result CurlEasyInit(CURL *& pCurlHandle)
{

    CURL *pCurlHandleTemporary = curl_easy_init();
    if (!pCurlHandleTemporary)
    {
        return ResultCurlEasyInitFailed();
    }

    pCurlHandle = pCurlHandleTemporary;
    return ResultSuccess();
}

void CurlEasyCleanup(CURL* pCurlHandle)
{
    NN_SDK_REQUIRES_NOT_NULL(pCurlHandle);

    curl_easy_cleanup(pCurlHandle);
}

void CurlEasySetOptWithoutResult(CURL* pCurlEasyHandle, CURLoption option, long value)
{
    NN_SDK_REQUIRES_NOT_NULL(pCurlEasyHandle);
    CURLcode code = curl_easy_setopt(pCurlEasyHandle, option, value);
    NN_SDK_ASSERT_EQUAL(code, CURLE_OK); NN_UNUSED(code);
}

template<typename T> void CurlEasySetOptWithoutResult(CURL* pCurlEasyHandle, CURLoption option, T value)
{
    NN_SDK_REQUIRES_NOT_NULL(pCurlEasyHandle);
    CURLcode code = curl_easy_setopt(pCurlEasyHandle, option, value);
    NN_SDK_ASSERT_EQUAL(code, CURLE_OK); NN_UNUSED(code);
}

Result CurlEasySetOptWithResult(CURL* pCurlEasyHandle, CURLoption option, const char * value)
{
    NN_SDK_REQUIRES_NOT_NULL(pCurlEasyHandle);
    CURLcode code = curl_easy_setopt(pCurlEasyHandle, option, value);
    return ConvertCurlCodeToResult(code);
}

Result ConvertCurlCodeToResult(CURLcode code)
{
    switch (code)
    {
    case CURLE_OK:
        return ResultSuccess();
    case CURLE_UNSUPPORTED_PROTOCOL:
        return ResultCurlErrorUnsupportedProtocol();
    case CURLE_FAILED_INIT:
        return ResultCurlErrorFailedInit();
    case CURLE_URL_MALFORMAT:
        return ResultCurlErrorUrlMalformat();
    case CURLE_NOT_BUILT_IN:
        return ResultCurlErrorNotBuiltIn();
    case CURLE_COULDNT_RESOLVE_PROXY:
        return ResultCurlErrorCouldntResolveProxy();
    case CURLE_COULDNT_RESOLVE_HOST:
        return ResultCurlErrorCouldntResolveHost();
    case CURLE_COULDNT_CONNECT:
        return ResultCurlErrorCouldntConnect();
    case CURLE_REMOTE_ACCESS_DENIED:
        return ResultCurlErrorRemoteAccessDenied();
    case CURLE_HTTP2:
        return ResultCurlErrorHttp2();
    case CURLE_PARTIAL_FILE:
        return ResultCurlErrorPartialFile();
    case CURLE_QUOTE_ERROR:
        return ResultCurlErrorQuoteError();
    case CURLE_HTTP_RETURNED_ERROR:
        return ResultCurlErrorHttpReturnedError();
    case CURLE_WRITE_ERROR:
        return ResultCurlErrorWriteError();
    case CURLE_UPLOAD_FAILED:
        return ResultCurlErrorUploadFailed();
    case CURLE_READ_ERROR:
        return ResultCurlErrorReadError();
    case CURLE_OUT_OF_MEMORY:
        return ResultCurlErrorOutOfMemory();
    case CURLE_OPERATION_TIMEDOUT:
        return ResultCurlErrorOperationTimedout();
    case CURLE_RANGE_ERROR:
        return ResultCurlErrorRangeError();
    case CURLE_HTTP_POST_ERROR:
        return ResultCurlErrorHttpPostError();
    case CURLE_SSL_CONNECT_ERROR:
        return ResultCurlErrorSslConnectError();
    case CURLE_BAD_DOWNLOAD_RESUME:
        return ResultCurlErrorBadDownloadResume();
    case CURLE_FUNCTION_NOT_FOUND:
        return ResultCurlErrorFunctionNotFound();
    case CURLE_ABORTED_BY_CALLBACK:
        return ResultCurlErrorAbortedByCallback();
    case CURLE_BAD_FUNCTION_ARGUMENT:
        return ResultCurlErrorBadFunctionArgument();
    case CURLE_INTERFACE_FAILED:
        return ResultCurlErrorInterfaceFailed();
    case CURLE_TOO_MANY_REDIRECTS:
        return ResultCurlErrorTooManyRedirects();
    case CURLE_UNKNOWN_OPTION:
        return ResultCurlErrorUnknownOption();
    case CURLE_PEER_FAILED_VERIFICATION:
        return ResultCurlErrorPeerFailedVerification();
    case CURLE_GOT_NOTHING:
        return ResultCurlErrorGotNothing();
    case CURLE_SSL_ENGINE_NOTFOUND:
        return ResultCurlErrorSslEngineNotFound();
    case CURLE_SSL_ENGINE_SETFAILED:
        return ResultCurlErrorSslEngineSetFailed();
    case CURLE_SEND_ERROR:
        return ResultCurlErrorSendError();
    case CURLE_RECV_ERROR:
        return ResultCurlErrorRecvError();
    case CURLE_SSL_CERTPROBLEM:
        return ResultCurlErrorSslCertProblem();
    case CURLE_SSL_CIPHER:
        return ResultCurlErrorSslCipher();
    case CURLE_SSL_CACERT:
        return ResultCurlErrorSslCaCert();
    case CURLE_BAD_CONTENT_ENCODING:
        return ResultCurlErrorBadContentEncoding();
    case CURLE_FILESIZE_EXCEEDED:
        return ResultCurlErrorFileSizeExceeded();
    case CURLE_USE_SSL_FAILED:
        return ResultCurlErrorUseSslFailed();
    case CURLE_SEND_FAIL_REWIND:
        return ResultCurlErrorSendFailRewind();
    case CURLE_SSL_ENGINE_INITFAILED:
        return ResultCurlErrorSslEngineInitFailed();
    case CURLE_LOGIN_DENIED:
        return ResultCurlErrorLoginDenied();
    case CURLE_CONV_FAILED:
        return ResultCurlErrorConvFailed();
    case CURLE_CONV_REQD:
        return ResultCurlErrorConvReqd();
    case CURLE_SSL_CACERT_BADFILE:
        return ResultCurlErrorSslCaCertBadFile();
    case CURLE_SSL_SHUTDOWN_FAILED:
        return ResultCurlErrorSslShutdownFailed();
    case CURLE_AGAIN:
        return ResultCurlErrorAgain();
    case CURLE_SSL_CRL_BADFILE:
        return ResultCurlErrorSslCrlBadFile();
    case CURLE_SSL_ISSUER_ERROR:
        return ResultCurlErrorSslIssuerError();
    case CURLE_CHUNK_FAILED:
        return ResultCurlErrorChunkFailed();
    case CURLE_NO_CONNECTION_AVAILABLE:
        return ResultCurlErrorNoConnectionAvailable();
    case CURLE_SSL_PINNEDPUBKEYNOTMATCH:
        return ResultCurlErrorSslPinnedPubkeyNotMatch();
    case CURLE_SSL_INVALIDCERTSTATUS:
        return ResultCurlErrorSslInvalidCertStatus();
    case CURLE_SSL_CTX_FUNCTION_NOT_SET:
        return ResultCurlErrorSslCtxFunctionNotSet();
    case CURLE_SSL_CTX_INVALID:
        return ResultCurlErrorSslCtxInvalid();
    case CURLE_SSL_CTX_FATAL:
        return ResultCurlErrorSslCtxFatal();
    case CURLE_SSL_ALREADY_CONNECTED:
        return ResultCurlErrorSslAlreadyConnected();
    case CURLE_SSL_INVALID_REFERENCE:
        return ResultCurlErrorSslInvalidReference();

    default:
        return ResultCurlErrorUnknown();
    }
} // NOLINT(impl/function_size)

Result ConvertCurlMultiCodeToResult(CURLMcode mcode)
{
    switch (mcode)
    {
    case CURLM_OK:
        return ResultSuccess();
    case CURLM_BAD_HANDLE:
        return ResultCurlMultiErrorBadHandle();
    case CURLM_BAD_EASY_HANDLE:
        return ResultCurlMultiErrorBadEasyHandle();
    case CURLM_OUT_OF_MEMORY:
        return ResultCurlMultiErrorOutOfMemory();
    case CURLM_INTERNAL_ERROR:
        return ResultCurlMultiErrorInternalError();
    case CURLM_BAD_SOCKET:
        return ResultCurlMultiErrorBadSocket();
    case CURLM_UNKNOWN_OPTION:
        return ResultCurlMultiErrorUnknownOption();
    case CURLM_ADDED_ALREADY:
        return ResultCurlMultiErrorAddedAlready();

    default:
        return ResultCurlMultiErrorUnknown();
    }
}

void SetAllocateFunctions(AllocateFunction allocfun, FreeFunction freefun, ReAllocateFunction reallocfun)
{
    s_AllocateFunctions = { allocfun, freefun, reallocfun };
}

void* AllocateMemory(size_t size)
{
    NN_SDK_ASSERT_NOT_NULL(s_AllocateFunctions.allocate);
    return s_AllocateFunctions.allocate(size);
}

void FreeMemory(void* p)
{
    NN_SDK_ASSERT_NOT_NULL(s_AllocateFunctions.free);
    return s_AllocateFunctions.free(p);
}

void* ReAllocateMemory(void* p, size_t size)
{
    NN_SDK_ASSERT_NOT_NULL(s_AllocateFunctions.realloc);
    return s_AllocateFunctions.realloc(p, size);
}

void* AllocateZeroInitializedArray(size_t number, size_t size)
{
    NN_SDK_ASSERT_NOT_NULL(s_AllocateFunctions.allocate);
    const size_t bytes = number*size;
    void* p = s_AllocateFunctions.allocate(bytes);
    if (!p)
    {
        return nullptr;
    }
    std::memset(p, 0, bytes);
    return p;
}

char *DuplicateString(const char *s)
{
    size_t l = strlen(s);
    void *d = AllocateMemory(l + 1);
    if (!d)
    {
        return nullptr;
    }
    return reinterpret_cast<char*>(std::memcpy(d, s, l + 1));
}

}} // ~namespace nn::http
