﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http.h>

#include <nn/nn_SdkLog.h>
#include <curl/curl.h>

#include <cctype>
#include <mutex>

#include "http_Utility.h"

#define NN_HTTP_SYNCHRONIZED std::lock_guard<os::Mutex> synchronized(m_mutex)

namespace nn {
namespace http {

ResponseImpl::ResponseImpl()
    : m_pCurlEasyHandle(nullptr)
    , m_pConnection(nullptr)
    , m_cond()
    , m_mutex(true)
    , m_mutexForCond(true)
    , m_pCancelEvent(nullptr)
    , m_CurlCode(CURLE_OK)
{
    Dispose();
}

ResponseImpl::~ResponseImpl()
{
    Dispose();
}

void ResponseImpl::Dispose()
{
    if (m_pConnection)
    {
        m_bCancelRequested = true;
        WaitCompleted();
    }

    m_bCancelRequested = false;
    m_State = ConnectionState_None;
    m_CurlCode = CURLE_OK;
    m_pCancelEvent = nullptr;

    if (m_pCurlEasyHandle)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pConnection);
        m_pConnection->CurlRemoveEasyHandleFromMulti(m_pCurlEasyHandle);
        curl_easy_cleanup(m_pCurlEasyHandle);
    }
    m_pCurlEasyHandle = nullptr;
}

int16_t ResponseImpl::GetStatusCode()
{
    long codeHttpStatus = -1;
    CURLcode code = curl_easy_getinfo(m_pCurlEasyHandle, CURLINFO_RESPONSE_CODE, &codeHttpStatus);
    NN_SDK_ASSERT_EQUAL(code, CURLE_OK); NN_UNUSED(code);
    return static_cast<int16_t>(codeHttpStatus);
}

Result ResponseImpl::GetStatusCodeAsResult()
{
    Result result;
    int16_t status = GetStatusCode();
    if (100 <= status && status <= 599)
    {
        auto innerValue = nn::result::detail::ResultTraits::MakeInnerValue(
            ResultHttpStatusContinue().GetModule(),
            status
        );
        result = nn::result::detail::ConstructResult(innerValue);
    }
    else
    {
        result = GetLastResult();
        if (result.IsSuccess())
        {
            result = ResultNotConnected();
        }
    }
    return result;
}

int64_t ResponseImpl::GetContentLength()
{
    double valueContentLength = -1;
    CURLcode code = curl_easy_getinfo(m_pCurlEasyHandle, CURLINFO_CONTENT_LENGTH_DOWNLOAD, &valueContentLength);
    NN_SDK_ASSERT_EQUAL(code, CURLE_OK); NN_UNUSED(code);
    return static_cast<int64_t>(valueContentLength);
}

const char * ResponseImpl::GetHeaderValue(const char * name)
{
    return nullptr;
}

ResponseImpl* ResponseImpl::GetInstanceFromEasyHandle(CURL* pCurlEasyHandle)
{
    char* pData = nullptr;
    CURLcode code = curl_easy_getinfo(pCurlEasyHandle, CURLINFO_PRIVATE, &pData);
    NN_SDK_ASSERT_EQUAL(code, CURLE_OK); NN_UNUSED(code);
    return reinterpret_cast<ResponseImpl*>(pData);
}

void ResponseImpl::OnHeaderReceived(const nn::util::string_view& name, const nn::util::string_view& value)
{
//    NN_SDK_LOG("(HEADER) %.*s: %.*s\n", name.length(), name.data(), value.length(), value.data());
}

void ResponseImpl::OnTransfer(curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow)
{
//    NN_SDK_LOG("dl %lld/%lld, ul %lld/%lld\n", dlNow, dlTotal, ulNow, ulTotal);
}

ConnectionState ResponseImpl::GetState() const
{
    return m_State;
}

void ResponseImpl::NotifyConnected()
{
    {
        std::lock_guard<os::Mutex> waitConditionBlocker(m_mutexForCond);

        if (m_State <= ConnectionState_Started)
        {
//            NN_SDK_LOG("State -> Connected\n");

            m_State = ConnectionState_SendingBody;
            m_cond.Broadcast();
        }
    }
}

void ResponseImpl::NotifyCompletedWithCode(CURLcode code)
{
    std::lock_guard<os::Mutex> waitConditionBlocker(m_mutexForCond);
    NN_SDK_ASSERT_GREATER_EQUAL(m_State, ConnectionState_Started);

    m_State = ConnectionState_Completed;
    m_CurlCode = code;
    //NN_SDK_LOG("NotifyCompletedWithCode: %s\n", curl_easy_strerror(m_CurlCode));
    m_cond.Broadcast();
}

Result ResponseImpl::GetLastResult() const
{
    if (m_CurlCode != CURLE_OK)
    {
        return ConvertCurlCodeToResult(m_CurlCode);
    }
    else if (m_bCancelRequested)
    {
        return ResultCancelled();
    }
    return ResultSuccess();
}

bool ResponseImpl::CheckPendingCancelRequest()
{
    if (m_pCancelEvent && m_pCancelEvent->TryWait())
    {
        NN_HTTP_INFO("Cancel requested.\n");
        m_bCancelRequested = true;
    }
    return m_bCancelRequested;
}

bool ResponseImpl::IsStarted() const
{
    return m_State >=ConnectionState_Started;
}

bool ResponseImpl::IsConnected() const
{
    return m_State >= ConnectionState_SendingBody;
}

bool ResponseImpl::IsCompleted() const
{
    return m_State >= ConnectionState_Completed;
}

bool ResponseImpl::WaitConnected()
{
    std::lock_guard<os::Mutex> waitConditionBlocker(m_mutexForCond);
    if (!IsStarted())
    {
        return false;
    }

    while (!IsConnected() && !CheckPendingCancelRequest())
    {
        m_pConnection->PsuedoWaitCondition(m_mutexForCond, m_cond, m_pCancelEvent);
    }
    return IsConnected();
}

bool ResponseImpl::WaitCompleted()
{
    std::lock_guard<os::Mutex> waitConditionBlocker(m_mutexForCond);
    if (!IsStarted())
    {
        return false;
    }

    while (!IsCompleted() && !CheckPendingCancelRequest())
    {
        m_pConnection->PsuedoWaitCondition(m_mutexForCond, m_cond, m_pCancelEvent);
    }
    return IsCompleted();
}

Result ResponseImpl::AssignHandle(CURL* pCurlEasyHandle, ConnectionBroker& connection)
{
    AttachResponseToEasyHandle(pCurlEasyHandle);

    {
        std::lock_guard<os::Mutex> waitConditionBlocker(m_mutexForCond);
        NN_SDK_ASSERT_EQUAL(m_State, ConnectionState_None);
        NN_SDK_ASSERT_EQUAL(m_pCurlEasyHandle, nullptr);

        m_pCurlEasyHandle = pCurlEasyHandle;
        m_pConnection = &connection;
        m_State = ConnectionState_Started;
    }
    return ResultSuccess();
}

void ResponseImpl::AttachResponseToEasyHandle(CURL * pCurlEasyHandle)
{
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_XFERINFOFUNCTION, CurlXferInfoFunction);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_XFERINFODATA,   static_cast<void*>(this));

    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_HEADERFUNCTION, CurlHeaderFunction);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_HEADERDATA,     static_cast<void*>(this));

    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_WRITEFUNCTION,  CurlWriteFunction);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_WRITEDATA,      static_cast<void*>(this));

    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_PRIVATE,        static_cast<void*>(this));
}


int ResponseImpl::CurlXferInfoFunction(void * clientp, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow)
{
    NN_UNUSED(dlTotal); NN_UNUSED(dlNow); NN_UNUSED(ulTotal); NN_UNUSED(ulNow);

    ResponseImpl* pThis = static_cast<ResponseImpl*>(clientp);

    pThis->OnTransfer(dlTotal, dlNow, ulTotal, ulNow);

    if (pThis->CheckPendingCancelRequest())
    {
        return CurlCallbackResult_Cancel;
    }
    return CurlCallbackResult_Continue;
}

size_t ResponseImpl::CurlHeaderFunction(char * pBuffer, size_t size, size_t nmemb, void * pUserdata)
{
    ResponseImpl* pThis = static_cast<ResponseImpl*>(pUserdata);
    size_t totalSize = size * nmemb;
    util::string_view line(pBuffer, totalSize), headerName, headerValue;

    // ヘッダの : を検索
    size_t posColon = line.find_first_of(":");
    if (posColon != util::string_view::npos && posColon > 1 && totalSize > posColon + 1)
    {
        // 改行を検索 (\r\n or \n)
        size_t posNewLine = line.find_first_of("\r", posColon + 1);
        if (posNewLine == util::string_view::npos)
        {
            posNewLine = line.find_first_of("\n", posColon + 1);
        }

        // : と 改行が見つかったらコールバックを呼び出す
        if (posNewLine != util::string_view::npos)
        {
            // ヘッダ値のスペースを読み飛ばす
            size_t posValue;
            for (posValue = posColon + 1; posValue < posNewLine; ++posValue)
            {
                if (!std::isspace(line.at(posValue)))
                {
                    break;
                }
            }

            // ヘッダと値の string_view を構築
            headerValue = line.substr(posValue, posNewLine - posValue);
            headerName  = line.substr(0, posColon);

            pThis->OnHeaderReceived(headerName, headerValue);
        }
    }

    return totalSize;
}

size_t ResponseImpl::CurlWriteFunction(char * pBuffer, size_t size, size_t nmemb, void * pUserdata)
{
    ResponseImpl* pThis = static_cast<ResponseImpl*>(pUserdata);
    size_t totalSize = size * nmemb;

    if (pThis->CheckPendingCancelRequest())
    {
        return 0;
    }

    pThis->NotifyConnected();

    //NN_SDK_LOG(" (BODY) %.*s\n", totalSize, pBuffer);

    if (!pThis->OnFillBuffer(pBuffer, totalSize))
    {
        return CURL_WRITEFUNC_PAUSE;
    }
    return totalSize;
}


}
} // ~namespace nn::http
