﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/*
 * @file 最大サイズが固定で、要素の追加・削除時にアロケーション操作を行わない map
 */

#include <algorithm>
#include <functional>
#include <iterator>
#include <utility>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

namespace nn { namespace htclow { namespace util {

    /*
    * @brief    最大サイズが固定の map
    * @detail   最大サイズが固定で、要素の追加・削除時にアロケーション操作を行わない map です。
    *           API は std::map ライクですが、htclow で必要な一部の機能しか実装していません。
    *           実装の都合上、要素の追加・削除時には Key と T のコピーが発生します。
    *           std::map で例外が投げられるような操作を行った場合や、その他想定外の操作を行った場合はアボートします。
    */
    template<size_t MaxSize, class Key, class T, class Compare = std::less<Key>>
    class FixedSizeMap
    {
    public:
        // メンバ型
        using key_type = Key;
        using value_type = std::pair<Key, T>;
        using mapped_type = T;
        using key_compare = Compare;
        using reference = value_type&;
        using const_reference = const value_type&;
        using iterator = value_type*;
        using const_iterator = const value_type*;
        using size_type = size_t;

        class value_compare
        {
        public:
            using result_type = bool;
            using first_argument_type = value_type;
            using second_argument_type = value_type;

            bool operator()(const_reference x, const_reference y) NN_NOEXCEPT
            {
                return comp(x.first, y.first);
            }

        protected:
            key_compare comp;
        };

        // std::lower_bound(), std::upper_bound(), std::equal_range() に渡すための前方向イテレータ
        // operator* で key の参照を返し、operator++ で次の要素に進む
        class _key_iterator
        {
        public:
            friend FixedSizeMap;

            using difference_type = ptrdiff_t;
            using value_type = key_type;
            using pointer = key_type*;
            using reference = key_type&;
            using iterator_category = std::forward_iterator_tag;

            _key_iterator() NN_NOEXCEPT
                : m_Impl(nullptr)
            {
            }

            _key_iterator(const _key_iterator& it) NN_NOEXCEPT
                : m_Impl(it.m_Impl)
            {
            }

            explicit _key_iterator(iterator it) NN_NOEXCEPT
                : m_Impl(it)
            {
            }

            reference operator*() const NN_NOEXCEPT
            {
                return m_Impl->first;
            }

            pointer operator->() const NN_NOEXCEPT
            {
                return &(m_Impl->first);
            }

            _key_iterator& operator++() NN_NOEXCEPT
            {
                m_Impl++;
                return *this;
            }

            _key_iterator& operator++(int) NN_NOEXCEPT
            {
                _key_iterator temporary(*this);
                ++(*this);
                return temporary;
            }

            bool operator==(const _key_iterator& target) const NN_NOEXCEPT
            {
                return m_Impl == target.m_Impl;
            }

            bool operator!=(const _key_iterator& target) const NN_NOEXCEPT
            {
                return m_Impl != target.m_Impl;
            }

        private:
            iterator m_Impl;
        };

        // _key_iterator の const 版
        class _const_key_iterator
        {
        public:
            friend FixedSizeMap;

            using difference_type = ptrdiff_t;
            using value_type = key_type;
            using pointer = const key_type*;
            using reference = const key_type&;
            using iterator_category = std::forward_iterator_tag;

            _const_key_iterator() NN_NOEXCEPT
                : m_Impl(nullptr)
            {
            }

            _const_key_iterator(const _const_key_iterator& it) NN_NOEXCEPT
                : m_Impl(it.m_Impl)
            {
            }

            explicit  _const_key_iterator(const_iterator it) NN_NOEXCEPT
                : m_Impl(it)
            {
            }

            reference operator*() const NN_NOEXCEPT
            {
                return m_Impl->first;
            }

            pointer operator->() const NN_NOEXCEPT
            {
                return &(m_Impl->first);
            }

            _const_key_iterator& operator++() NN_NOEXCEPT
            {
                m_Impl++;
                return *this;
            }

            _const_key_iterator& operator++(int) NN_NOEXCEPT
            {
                _const_key_iterator temporary(*this);
                ++(*this);
                return temporary;
            }

            bool operator==(const _const_key_iterator& target) const NN_NOEXCEPT
            {
                return m_Impl == target.m_Impl;
            }

            bool operator!=(const _const_key_iterator& target) const NN_NOEXCEPT
            {
                return m_Impl != target.m_Impl;
            }

        private:
            const_iterator m_Impl;
        };

        // メンバ関数
        FixedSizeMap() NN_NOEXCEPT
            : m_Count(0)
        {
        }

        iterator begin() NN_NOEXCEPT
        {
            return &m_Array[0];
        }

        const_iterator begin() const NN_NOEXCEPT
        {
            return &m_Array[0];
        }

        const_iterator cbegin() const NN_NOEXCEPT
        {
            return &m_Array[0];
        }

        iterator end() NN_NOEXCEPT
        {
            return &m_Array[m_Count];
        }

        const_iterator end() const NN_NOEXCEPT
        {
            return &m_Array[m_Count];
        }

        const_iterator cend() const NN_NOEXCEPT
        {
            return &m_Array[m_Count];
        }

        bool empty() const NN_NOEXCEPT
        {
            return m_Count == 0;
        }

        size_type size() const NN_NOEXCEPT
        {
            return m_Count;
        }

        size_type max_size() const NN_NOEXCEPT
        {
            return MaxSize;
        }

        std::pair<iterator, bool> insert(const value_type& x) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_Count < MaxSize);

            if (count(x.first) > 0)
            {
                return std::make_pair(end(), false);
            }

            // 挿入位置より後ろの要素を1つずつずらしてから、挿入
            iterator insertPosition = upper_bound(x.first);
            for (auto it = end(); it != insertPosition; --it)
            {
                *it = *(it - 1);
            }
            *insertPosition = x;

            m_Count++;
            return std::make_pair(insertPosition, true);
        }

        // 要素を削除するが、std::map と異なりデストラクタは呼び出さない
        void erase(iterator position) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_Count > 0);
            NN_ABORT_UNLESS(begin() <= position);
            NN_ABORT_UNLESS(position < end());

            // 削除して1つずつ詰める
            for (auto it = position; it < end() - 1; it++)
            {
                *it = *(it + 1);
            }

            m_Count--;
        }

        void erase(const key_type& x) NN_NOEXCEPT
        {
            erase(find(x));
        }

        T& at(const key_type& x) NN_NOEXCEPT
        {
            auto first = _key_iterator(begin());
            auto last = _key_iterator(end());
            auto keyIter = std::lower_bound(first, last, x, key_comp());

            if (keyIter == last)
            {
                NN_ABORT();
            }

            NN_ABORT_UNLESS(keyIter != last);
            NN_ABORT_UNLESS(keyIter.m_Impl->first == x);

            return keyIter.m_Impl->second;
        }

        const T& at(const key_type& x) const NN_NOEXCEPT
        {
            auto first = _key_iterator(begin());
            auto last = _key_iterator(end());
            auto keyIter = std::lower_bound(first, last, x, key_comp());

            NN_ABORT_UNLESS(keyIter != last);
            NN_ABORT_UNLESS(keyIter.m_Impl->first == x);

            return keyIter.m_Impl->second;
        }

        size_type count(const key_type& x) const NN_NOEXCEPT
        {
            auto it = lower_bound(x);

            if (it != end() && it->first == x)
            {
                return 1;
            }
            else
            {
                return 0;
            }
        }

        iterator find(const key_type& x) NN_NOEXCEPT
        {
            auto it = lower_bound(x);

            if (it != end() && it->first == x)
            {
                return it;
            }
            else
            {
                return end();
            }
        }

        const_iterator find(const key_type& x) const NN_NOEXCEPT
        {
            auto it = lower_bound(x);

            if (it != end() && it->first == x)
            {
                return it;
            }
            else
            {
                return end();
            }
        }

        iterator equal_range(const key_type& x) NN_NOEXCEPT
        {
            return std::equal_range(_key_iterator(begin()), _key_iterator(end()), x, key_comp()).m_Impl;
        }

        const_iterator equal_range(const key_type& x) const NN_NOEXCEPT
        {
            return std::equal_range(_const_key_iterator(cbegin()), _const_key_iterator(cend()), x, key_comp()).m_Impl;
        }

        iterator lower_bound(const key_type& x) NN_NOEXCEPT
        {
            return std::lower_bound(_key_iterator(begin()), _key_iterator(end()), x, key_comp()).m_Impl;
        }

        const_iterator lower_bound(const key_type& x) const NN_NOEXCEPT
        {
            return std::lower_bound(_const_key_iterator(cbegin()), _const_key_iterator(cend()), x, key_comp()).m_Impl;
        }

        iterator upper_bound(const key_type& x) NN_NOEXCEPT
        {
            return std::upper_bound(_key_iterator(begin()), _key_iterator(end()), x, key_comp()).m_Impl;
        }

        const_iterator upper_bound(const key_type& x) const NN_NOEXCEPT
        {
            return std::upper_bound(_const_key_iterator(cbegin()), _const_key_iterator(cend()), x, key_comp()).m_Impl;
        }

        key_compare key_comp() const NN_NOEXCEPT
        {
            return key_compare();
        }

        value_compare value_comp() const NN_NOEXCEPT
        {
            return value_compare();
        }

    private:
        // 要素を格納する配列
        value_type m_Array[MaxSize];

        // 有効な要素の数
        size_t m_Count;
    };

}}}
