﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <numeric>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/htclow/htclow_ResultPrivate.h>

#include "htclow_SendBuffer.h"

namespace nn { namespace htclow { namespace server { namespace mux {

SendBuffer::SendBuffer(bool retransmitEnabled) NN_NOEXCEPT
    : m_EmptyEvent(nn::os::EventClearMode_ManualClear, true)
    , m_NextSequenceId(0)
    , m_RetransmitEnabled(retransmitEnabled)
{
}

SendBuffer::~SendBuffer() NN_NOEXCEPT
{
    while (!m_Packets.empty())
    {
        auto pFront = &m_Packets.front();
        m_Packets.pop_front();
        delete pFront;
    }

    while (!m_AckPackets.empty())
    {
        auto pFront = &m_AckPackets.front();
        m_AckPackets.pop_front();
        delete pFront;
    }
}

Result SendBuffer::AddPacket(std::unique_ptr<SendPacket> packet) NN_NOEXCEPT
{
    const auto packetType = packet->GetHeader()->packetType;

    if (IsAckPacket(packetType))
    {
        return ResultInvalidArgument();
    }

    if (packet->GetHeader()->packetType == PacketType::Syn && m_NextSequenceId != 0)
    {
        return ResultChannelSequenceIdNotMatched();
    }

    packet->GetHeader()->sequenceId = m_NextSequenceId;
    m_NextSequenceId++;

    m_EmptyEvent.Clear();

    // リストに繋ぐ場合は packet は手動で delete
    auto p = packet.release();

    m_Packets.push_back(*p);
    NN_RESULT_SUCCESS;
}

Result SendBuffer::AddAckPacket(std::unique_ptr<SendPacket> packet) NN_NOEXCEPT
{
    const auto packetType = packet->GetHeader()->packetType;

    if (!IsAckPacket(packetType))
    {
        return ResultInvalidArgument();
    }

    m_EmptyEvent.Clear();

    // リストに繋ぐ場合は packet は手動で delete
    auto p = packet.release();

    m_AckPackets.push_back(*p);
    NN_RESULT_SUCCESS;
}

void SendBuffer::RemovePacket(int64_t sequenceId) NN_NOEXCEPT
{
    int removeCount = 0;
    for (auto& packet: m_Packets)
    {
        if (packet.GetHeader()->sequenceId <= sequenceId)
        {
            removeCount++;
        }
    }

    for (int i = 0; i < removeCount; i++)
    {
        auto pFront = &m_Packets.front();
        m_Packets.pop_front();
        delete pFront;
    }

    if (m_Packets.empty() && m_AckPackets.empty())
    {
        m_EmptyEvent.Signal();
    }
}

void SendBuffer::RemoveAckPacket(int64_t sequenceId) NN_NOEXCEPT
{
    int removeCount = 0;
    for (auto& packet: m_AckPackets)
    {
        if (packet.GetHeader()->sequenceId <= sequenceId)
        {
            removeCount++;
        }
    }

    for (int i = 0; i < removeCount; i++)
    {
        auto pFront = &m_AckPackets.front();
        m_AckPackets.pop_front();
        delete pFront;
    }

    if (m_Packets.empty() && m_AckPackets.empty())
    {
        m_EmptyEvent.Signal();
    }
}

SendPacket* SendBuffer::GetNextPacket() NN_NOEXCEPT
{
    // 送るべき ack があれば、まずそれを送る
    if (!m_AckPackets.empty())
    {
        return &m_AckPackets.front();
    }

    // ack 以外のパケットのリストが空なら、以後の処理はしない
    if (m_Packets.empty())
    {
        return nullptr;
    }

    UpdateForRetransmit();

    // 未送信パケットのうち最初のものを取得
    auto iterToSend = std::find_if(m_Packets.begin(), m_Packets.end(), [](auto& node) NN_NOEXCEPT
    {
        return !node.GetSendFlag();
    });
    if (iterToSend == m_Packets.end())
    {
        return nullptr;
    }

    // 送信サイズの合計がスライディングウィンドウサイズを超えてしまうなら送信しない
    int sendingPacketCount = 0;
    for (auto iter = m_Packets.begin(); iter != iterToSend; iter++)
    {
        ++sendingPacketCount;
    }

    if (sendingPacketCount >= MaxSendingPacketCount)
    {
        return nullptr;
    }

    // パケットの送信が決定
    iterToSend->SetSendFlag(true);
    iterToSend->SetSendTime(nn::os::GetSystemTick());

    return &(*iterToSend);
}

void SendBuffer::UpdateForRetransmit() NN_NOEXCEPT
{
    if (!m_RetransmitEnabled)
    {
        return;
    }

    // リストの先頭のパケットの再送時刻になったら、すべてのパケットの送信履歴を破棄
    const auto& front = m_Packets.front();

    const auto now = nn::os::GetSystemTick();
    if (front.GetSendFlag() && nn::os::ConvertToTimeSpan(now - front.GetSendTime()) > RetransmitTimeout)
    {
        for (auto& packet : m_Packets)
        {
            packet.SetSendFlag(false);
            packet.SetSendTime(nn::os::Tick(0));
        }
    }
}

}}}}
