﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/htclow/htclow_ResultPrivate.h>

#include "htclow_ReceiveBuffer.h"

namespace nn { namespace htclow { namespace server { namespace mux {

ReceiveBuffer::ReceiveBuffer() NN_NOEXCEPT
    : m_ReceiveReadyEvent(nn::os::EventClearMode_ManualClear, true)
{
}

ReceiveBuffer::~ReceiveBuffer() NN_NOEXCEPT
{
    while (!m_Packets.empty())
    {
        auto pFront = &m_Packets.front();
        m_Packets.pop_front();
        delete pFront;
    }
}

Result ReceiveBuffer::AddPacket(std::unique_ptr<ReceivePacket> packet) NN_NOEXCEPT
{
    auto pHeader = packet->GetHeader();

    if (m_NextSequenceId != pHeader->sequenceId)
    {
        // パケットを破棄
        return ResultChannelSequenceIdNotMatched();
    }

    auto readbleBodySize = GetReadableBodySize();
    if (readbleBodySize + pHeader->bodySize > MaxChannelBufferSize)
    {
        return ResultChannelBufferOverflow();
    }

    if (pHeader->bodySize > 0)
    {
        // リストに繋ぐ場合、ReceivePacket の delete は手動で行う
        ReceivePacket* p = packet.release();

        m_Packets.push_back(*p);
        m_ReceiveReadyEvent.Signal();
    }
    m_NextSequenceId++;

    NN_RESULT_SUCCESS;
}

int ReceiveBuffer::GetReadableBodySize() NN_NOEXCEPT
{
    // sum は MaxChannelBufferSize を超えないのでオーバーフローしない
    int sum = 0;
    for (auto& packet : m_Packets)
    {
        sum += packet.GetBodySize();
    }

    return sum - m_HeadOffset;
}

Result ReceiveBuffer::ReadBody(void* pOutBuffer, int bufferSize) NN_NOEXCEPT
{
    auto buffer = reinterpret_cast<char*>(pOutBuffer);

    // 異常系チェック
    if (bufferSize < 0)
    {
        return ResultInvalidArgument();
    }

    if (bufferSize == 0)
    {
        NN_RESULT_SUCCESS;
    }

    if (GetReadableBodySize() < bufferSize)
    {
        return ResultChannelBufferHasNotEnoughData();
    }

    // 正常系
    int totalCopiedSize = 0;
    while (totalCopiedSize < bufferSize)
    {
        auto& packet = m_Packets.front();
        PacketHeader* header = packet.GetHeader();
        uint8_t* body = packet.GetBody();

        int copySize = std::min(bufferSize - totalCopiedSize, header->bodySize - m_HeadOffset);

        std::memcpy(&buffer[totalCopiedSize], &body[m_HeadOffset], copySize);
        totalCopiedSize += copySize;

        m_HeadOffset += copySize;
        NN_SDK_ASSERT(m_HeadOffset <= header->bodySize);

        if (m_HeadOffset == header->bodySize)
        {
            // リストの先頭のパケットを最後まで読み切った場合
            m_Packets.pop_front();
            delete &packet;
            m_HeadOffset = 0;
        }
    }

    // すべてのデータを読み切った場合は m_ReceiveReadyEvent をクリア
    if (m_Packets.empty())
    {
        m_ReceiveReadyEvent.Clear();
    }

    NN_RESULT_SUCCESS;
}

}}}}
