﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>

#include <nn/socket.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/htclow.h>
#include <nn/htclow/htclow_ResultPrivate.h>
#include <nn/htclow/detail/htclow_Log.h>

#include "htclow_SocketDriver.h"

namespace nn { namespace htclow { namespace server { namespace driver {

Result SocketDriver::Open() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

void SocketDriver::Close() NN_NOEXCEPT
{
}

Result SocketDriver::Connect() NN_NOEXCEPT
{
    if (m_Connected == true)
    {
        NN_ABORT();
    }

    const int serverSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    if (serverSocket == nn::socket::InvalidSocket)
    {
        return ResultSocketAllocationFailure();
    }

    nn::socket::SockAddrIn serverAddress;
    std::memset(&serverAddress, 0, sizeof(serverAddress));
    serverAddress.sin_family = nn::socket::Family::Af_Inet;
    serverAddress.sin_port  = nn::socket::InetHtons(HtclowListenPort);
    serverAddress.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);

    const int bindResult = nn::socket::Bind(serverSocket, reinterpret_cast<nn::socket::SockAddr*>(&serverAddress), sizeof(serverAddress));
    if (bindResult != 0)
    {
        NN_DETAIL_HTCLOW_WARN("Bind() failed. (%d)\n", nn::socket::GetLastError());
        nn::socket::Close(serverSocket);
        return ResultSocketBindFailure();
    }

    const int listenResult = nn::socket::Listen(serverSocket, 1);
    if (listenResult != 0)
    {
        NN_DETAIL_HTCLOW_WARN("Listen() failed. (%d)\n", nn::socket::GetLastError());
        nn::socket::Close(serverSocket);
        return ResultSocketListenFailure();
    }

    nn::socket::SockAddrIn clientAddress;
    nn::socket::SockLenT clientAddressLength = sizeof(clientAddress);

    const int acceptResult = nn::socket::Accept(serverSocket, reinterpret_cast<nn::socket::SockAddr*>(&clientAddress), &clientAddressLength);
    if (acceptResult < 0)
    {
        NN_DETAIL_HTCLOW_WARN("Accept() failed. (%d)\n", nn::socket::GetLastError());
        nn::socket::Close(serverSocket);
        return ResultSocketAcceptFailure();
    }

    m_Socket = acceptResult;
    m_ServerSocket = serverSocket;
    m_Connected = true;

    NN_RESULT_SUCCESS;
}

void SocketDriver::Shutdown() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Connected);

    nn::socket::Shutdown(m_Socket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Close(m_Socket);

    nn::socket::Shutdown(m_ServerSocket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Close(m_ServerSocket);

    m_Connected = false;
}

Result SocketDriver::Send(const void* pBuffer, int bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Connected);

    if (bufferSize < 0)
    {
        return ResultInvalidArgument();
    }
    if (bufferSize == 0)
    {
        NN_RESULT_SUCCESS;
    }

    int totalSendSize = 0;
    while (totalSendSize < bufferSize)
    {
        auto buffer = reinterpret_cast<const uint8_t*>(pBuffer);
        const auto n = nn::socket::Send(m_Socket, &buffer[totalSendSize], bufferSize - totalSendSize, 0);

        if (n <= 0)
        {
            return ResultSocketSendFailure();
        }

        // n <= bufferSize なので問題ないキャスト
        totalSendSize += static_cast<int>(n);
    }

    NN_RESULT_SUCCESS;
}

Result SocketDriver::Receive(void* pOutBuffer, int bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Connected);

    if (bufferSize < 0)
    {
        return ResultInvalidArgument();
    }
    if (bufferSize == 0)
    {
        NN_RESULT_SUCCESS;
    }

    int totalReceiveSize = 0;
    while (totalReceiveSize < bufferSize)
    {
        auto buffer = reinterpret_cast<uint8_t*>(pOutBuffer);
        const auto n = nn::socket::Recv(m_Socket, &buffer[totalReceiveSize], bufferSize - totalReceiveSize, 0);

        if (n <= 0)
        {
            return ResultSocketReceiveFailure();
        }

        // n <= bufferSize なので問題ないキャスト
        totalReceiveSize += static_cast<int>(n);
    }

    NN_RESULT_SUCCESS;
}

void SocketDriver::Cancel() NN_NOEXCEPT
{
    nn::socket::Shutdown(m_Socket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Close(m_Socket);
}

}}}}
