﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/os.h>
#include "hdcp_Switch.h"
#include "detail/hdcp_HdcpCore.h"

#include <nn/hdcp/detail/hdcp_Log.h>

namespace nn {
namespace hdcp {
namespace impl {

namespace {

const int InitializeCountMax = 128;         // 初期化回数の上限値(カウントのオーバーフロー対策)
int g_InitializeCount;                      // 初期化回数のカウント
detail::HdcpCore g_Hdcp;                    // HDCPを実際にかけるインスタンス
HdcpMode g_CurrentHdcpMode;                 // HDCPが有効/無効かどうかを示す設定値

} // namespace


Result Initialize() NN_NOEXCEPT
{
    if (g_InitializeCount < InitializeCountMax)
    {
        ++g_InitializeCount;
        if (1 == g_InitializeCount)
        {
            return g_Hdcp.Open();
        }
    }
    // 初期化回数でASSERTには引っ掛けない。
    return nn::ResultSuccess();
}

void Finalize() NN_NOEXCEPT
{
    if (0 < g_InitializeCount)
    {
        --g_InitializeCount;
        if (0 == g_InitializeCount)
        {
            g_Hdcp.Close();
        }
    }
    // 初期化回数でASSERTには引っ掛けない。
}

Result GetHdcpStateTransitionEvent(nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    return g_Hdcp.GetHdcpStateTransitionEvent(pOutHandle);
}

Result GetHdcpAuthenticationState(HdcpAuthenticationState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    return g_Hdcp.GetCurrentState(pOutState);
}

Result SetCurrentHdcpMode(HdcpMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    auto result = g_Hdcp.SetHdcpConfig(mode);
    if (result.IsSuccess())
    {
        g_CurrentHdcpMode = mode;
    }
    return result;
}

HdcpMode GetCurrentHdcpMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(0, g_InitializeCount);
    return g_CurrentHdcpMode;
}

} // impl
} // hdcp
} // nn
