﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/hdcp/hdcp.h>
#include <nn/hdcp/hdcp_Result.h>
#include <nn/hdcp/hdcp_Type.h>

#include <nn/hdcp/detail/hdcp_IHdcpController.h>
#include "detail/hdcp_ServiceType.h"

#include "hdcp_CreateHdcpControllerByHipc.h"
#include "server/hdcp_CreateHdcpControllerByDfc.h"

#include <nn/nn_SdkLog.h>

// 本ライブラリは現在、同一プロセスで利用される想定。
#define NN_HDCP_ACCESS_BY_HIPC

namespace nn { namespace hdcp {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::hdcp::detail::IHdcpController> g_HdcpController;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
nn::os::Mutex g_InitializeCountMutex(false);

}

void Initialize(nn::sf::SharedPointer<nn::hdcp::detail::IHdcpController> pServiceObject) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(!g_HdcpController);

        // マクロによって、HIPC/DFC を切り替える
        #ifdef NN_HDCP_ACCESS_BY_HIPC
            g_HdcpController = pServiceObject;
        #else
            g_HdcpController = nn::hdcp::CreateHdcpControllerByDfc();
        #endif
    }

    g_InitializeCount++;
}

// 以下は DFC と HIPC とで共通コード
void Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {

        NN_SDK_ASSERT(g_HdcpController);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_HdcpController = nullptr;
    }
}

Result GetHdcpStateTransitionEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    nn::sf::NativeHandle sfHandle;

    NN_RESULT_DO(g_HdcpController->GetHdcpStateTransitionEvent(&sfHandle));

    // 取得したハンドルをイベントに関連付ける
    nn::os::AttachReadableHandleToSystemEvent(
        pEvent,
        sfHandle.GetOsHandle(),
        sfHandle.IsManaged(),
        nn::os::EventClearMode_AutoClear);

    // 渡されたイベントにハンドルの管理を移管したので、管理権を放棄
    sfHandle.Detach();

    NN_RESULT_SUCCESS;
}

Result GetHdcpAuthenticationState(HdcpAuthenticationState* pOutState) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_HdcpController);
    std::int32_t state;
    if (pOutState)
    {
        auto result = g_HdcpController->GetHdcpAuthenticationState(&state);
        *pOutState = static_cast<HdcpAuthenticationState>(state);
        return result;
    }
    // NG case.
    return g_HdcpController->GetHdcpAuthenticationState(nullptr);
}

Result SetCurrentHdcpMode(HdcpMode mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_HdcpController);
    return g_HdcpController->SetCurrentHdcpMode(mode);
}

HdcpMode GetCurrentHdcpMode() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_HdcpController);
    return static_cast<HdcpMode>(g_HdcpController->GetCurrentHdcpMode());
}

}}
