﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/gpio/gpio_Result.h>
#include <nn/gpio/detail/gpio_Log.h>

#include <nn/gpio/driver/gpio_IGpioDriver.h>
#include <nn/gpio/driver/gpio_Pad.h>
#include <nn/gpio/driver/detail/gpio_PadSessionImpl.h>

namespace nn {
namespace gpio {
namespace driver {
namespace detail {

PadSessionImpl::PadSessionImpl() NN_NOEXCEPT :
    m_EventHolder()
{
}

PadSessionImpl::~PadSessionImpl() NN_NOEXCEPT
{
    Close();
}

nn::Result PadSessionImpl::Open(Pad* pPad, nn::ddsf::AccessMode accessMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pPad);

    bool needsRollback = true;
    bool isFirstSessionOnPad = !pPad->HasAnyOpenSession();

    // 発見したパッドオブジェクトをセッションと相互に紐づけ
    NN_RESULT_DO(nn::ddsf::OpenSession(pPad, this, accessMode));
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            nn::ddsf::CloseSession(this);
        }
    };

    auto& driver = pPad->GetDriver().SafeCastTo<IGpioDriver>();
    if ( isFirstSessionOnPad )
    {
        NN_RESULT_DO(driver.InitializePad(pPad));
    }
    NN_UTIL_SCOPE_EXIT
    {
        if ( needsRollback && isFirstSessionOnPad )
        {
            driver.FinalizePad(pPad);
        }
    };

    needsRollback = false; // 成功
    NN_RESULT_SUCCESS;
}

void PadSessionImpl::Close() NN_NOEXCEPT
{
    // 多重クローズは許容
    if (!IsOpen())
    {
        return;
    }

    // Unbind されていないのであれば、Unbind する
    if (IsInterruptBound())
    {
        UnbindInterrupt();
    }

    auto& pad = GetDevice().SafeCastTo<Pad>(); // GetDevice() は Close 前でないと呼べないので注意

    // パッドオブジェクトとセッションの紐づけ解除
    nn::ddsf::CloseSession(this);

    // 最後のセッションだった場合は FinalizePad を呼ぶ
    if ( !pad.HasAnyOpenSession() )
    {
        auto& driver = pad.GetDriver().SafeCastTo<IGpioDriver>();
        driver.FinalizePad(&pad);
    }
}

nn::Result PadSessionImpl::BindInterrupt(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    bool needsRollback = true;

    auto& pad = GetDevice().SafeCastTo<Pad>();
    auto& interruptControlMutex = pad.GetDriver().SafeCastTo<IGpioDriver>().GetInterruptControlMutex(pad);
    std::lock_guard<decltype(interruptControlMutex)> lock(interruptControlMutex);

    // このセッション自身がすでにバインド済みならエラー
    if ( IsInterruptBound() )
    {
        return nn::gpio::ResultAlreadyBound();
    }

    // 複数クライアントが同一パッドにイベントを登録するのは非サポート（誰が割り込みをクリアしたり再設定したりするかの責任が解決できないため）
    if ( GetDevice().SafeCastTo<Pad>().IsAnySessionBoundToInterrupt() )
    {
        return nn::gpio::ResultAlreadyBound();
    }

    // SystemEvent を初期化する。
    // DFC で使う場合で最適化したい場合は、プロセス内通信として初期化した方がよい。
    NN_RESULT_DO(nn::os::CreateSystemEvent(pEvent, nn::os::EventClearMode_ManualClear, true));
    NN_UTIL_SCOPE_EXIT
    {
        if ( needsRollback )
        {
            nn::os::DestroySystemEvent(pEvent);
        }
    };

    // イベントをセッション内に登録
    m_EventHolder.AttachEvent(pEvent);
    NN_UTIL_SCOPE_EXIT
    {
        if ( needsRollback )
        {
            m_EventHolder.DetachEvent();
        }
    };

    NN_RESULT_DO(UpdateInterruptEnabledOnDriver());

    needsRollback = false; // 成功
    NN_RESULT_SUCCESS;
}

void PadSessionImpl::UnbindInterrupt() NN_NOEXCEPT
{
    auto& pad = GetDevice().SafeCastTo<Pad>();
    auto& interruptControlMutex = pad.GetDriver().SafeCastTo<IGpioDriver>().GetInterruptControlMutex(pad);
    std::lock_guard<decltype(interruptControlMutex)> lock(interruptControlMutex);

    // Bind 済みかチェック
    if ( !IsInterruptBound() )
    {
        NN_DETAIL_GPIO_WARN("Interrupt was already unbound\n");
        return;
    }

    // 渡されたイベントを外して破棄する。
    auto pDetachedEvent = m_EventHolder.DetachEvent();
    nn::os::DestroySystemEvent(pDetachedEvent);

    // この間に割り込みが入った場合、割り込みに対応するシステムイベントは見つからないが内部的にはフラグがクリアされるよう
    // ドライバが注意して実装しないといけない

    // レジスタ側の割り込みが無効化される
    NN_ABORT_UNLESS_RESULT_SUCCESS(UpdateInterruptEnabledOnDriver());
}

nn::Result PadSessionImpl::GetInterruptEnabled(bool* pOut) const NN_NOEXCEPT
{
    *pOut = GetDevice().SafeCastTo<Pad>().IsInterruptEnabled();

    NN_RESULT_SUCCESS;
}

nn::Result PadSessionImpl::SetInterruptEnabled(bool enable) NN_NOEXCEPT
{
    bool needsRollback = true;

    auto& pad = GetDevice().SafeCastTo<Pad>();
    auto& interruptControlMutex = pad.GetDriver().SafeCastTo<IGpioDriver>().GetInterruptControlMutex(pad);
    std::lock_guard<decltype(interruptControlMutex)> lock(interruptControlMutex);

    // パッドの割り込み有効状態を更新
    bool wasEnabled = pad.IsInterruptEnabled();
    pad.SetInterruptEnabled(enable);
    NN_UTIL_SCOPE_EXIT
    {
        if ( needsRollback )
        {
            pad.SetInterruptEnabled(wasEnabled);
        }
    };

    NN_RESULT_DO(UpdateInterruptEnabledOnDriver());

    needsRollback = false; // 成功
    NN_RESULT_SUCCESS;
}

void PadSessionImpl::SignalInterruptBoundEvent() NN_NOEXCEPT
{
    auto& pad = GetDevice().SafeCastTo<Pad>();
    auto& driver = pad.GetDriver().SafeCastTo<IGpioDriver>();
    NN_UNUSED(driver); // Release ビルド対策
    NN_SDK_ASSERT(driver.GetInterruptControlMutex(pad).IsLockedByCurrentThread(),
        "PadSessionImpl::SignalInterruptBoundEvent() must be called after obtaining lock for interrupt enable mutex of the pad.\n");

    auto pEvent = m_EventHolder.GetSystemEvent();
    if ( pEvent )
    {
        nn::os::SignalSystemEvent(pEvent);
    }
}

nn::Result PadSessionImpl::UpdateInterruptEnabledOnDriver() NN_NOEXCEPT
{
    auto& pad = GetDevice().SafeCastTo<Pad>();
    auto& driver = pad.GetDriver().SafeCastTo<IGpioDriver>();
    NN_SDK_ASSERT(driver.GetInterruptControlMutex(pad).IsLockedByCurrentThread(),
        "PadSessionImpl::UpdateInterruptEnabledOnDriver() must be called after obtaining lock for interrupt enable mutex of the pad.\n");

    // 内部の割り込みフラグが enable かつ Bind 済みならドライバに指示して割り込みを enable にする
    return driver.SetInterruptEnabled(&pad, pad.IsInterruptRequiredEnabledOnDriver());
}

} // detail
} // driver
} // gpio
} // nn
