﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h> // For InterruptThread
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/ddsf/ddsf_EventHandlerLooper.h>
#include <nn/ddsf/ddsf_DeviceCodeEntryManager.h>

#include <nn/gpio/detail/gpio_Log.h>
#include <nn/gpio/gpio_Result.h>

#include <nn/gpio/driver/gpio_IGpioDriver.h>
#include <nn/gpio/driver/gpio_Pad.h>
#include <nn/gpio/driver/detail/dt/gpio_DeviceCodeNodeParser.h>

#include "gpio_Core.h"

NN_DDSF_CAST_SAFE_DEFINE(nn::gpio::driver::IGpioDriver, nn::ddsf::IDriver);
NN_DDSF_CAST_SAFE_DEFINE(nn::gpio::driver::Pad, nn::ddsf::IDevice);
NN_DDSF_CAST_SAFE_DEFINE(nn::gpio::driver::detail::PadSessionImpl, nn::ddsf::ISession);

namespace {

nn::gpio::driver::IGpioDriver::List g_GpioDriverList;
nn::gpio::driver::detail::dt::DeviceCodeNodeRootParser g_GpioDeviceCodeNodeRootParser("subsys-gpio");
std::array<nn::ddsf::DeviceCodeEntryHolder,256> g_GpioDeviceCodeEntryBuffer;
nn::ddsf::DeviceCodeEntryManager g_GpioDeviceCodeEntryManager(&g_GpioDeviceCodeEntryBuffer);

// 割り込みハンドラスレッドに与えるスタックリソース
const size_t InterruptThreadStackSize = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char g_InterruptThreadStack[InterruptThreadStackSize];
nn::os::ThreadType g_InterruptThread;
nn::ddsf::EventHandlerLooper g_InterruptThreadLooper;

enum class DriverInitializationState
{
    NotInitialized,
    Initializing,
    Initialized,
    Finalizing
};

DriverInitializationState g_DriverInitializationState = DriverInitializationState::NotInitialized;

}

namespace nn { namespace gpio { namespace driver { namespace detail {

void InitializeDrivers() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(g_DriverInitializationState, DriverInitializationState::NotInitialized);

    g_DriverInitializationState = DriverInitializationState::Initializing;

    g_InterruptThreadLooper.Initialize();

    for ( auto& driver : g_GpioDriverList )
    {
        driver.SafeCastTo<IGpioDriver>().InitializeDriver();
    }

    // デバイスコードノードのパース実施
    g_GpioDeviceCodeNodeRootParser.ParseAll();

    // イベント通知スレッドを作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_InterruptThread,
            [](void* p) NN_NOEXCEPT
            {
                NN_UNUSED(p);
                g_InterruptThreadLooper.LoopAuto();
            },
            nullptr,
            g_InterruptThreadStack, InterruptThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(gpio, InterruptHandler))
        );
    nn::os::SetThreadNamePointer(&g_InterruptThread, NN_SYSTEM_THREAD_NAME(gpio, InterruptHandler));
    nn::os::StartThread(&g_InterruptThread);
    g_InterruptThreadLooper.WaitLoopEnter(); // LoopAuto に入ったことを保証

    g_DriverInitializationState = DriverInitializationState::Initialized;
}

void FinalizeDrivers() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(g_DriverInitializationState, DriverInitializationState::Initialized);

    g_DriverInitializationState = DriverInitializationState::Finalizing;

    // TORIAEZU: テスト環境で繰り返し初期化可能にするための実装なので、使用中セッションのクリーンアップなど行儀の良いことはせず、強制的にリソースを破棄していく

    g_InterruptThreadLooper.RequestStop();
    // g_InterruptThreadLooper.WaitLoopExit(); // 下でスレッド破棄を待機しているので十分
    nn::os::WaitThread(&g_InterruptThread);
    nn::os::DestroyThread(&g_InterruptThread);

    g_GpioDeviceCodeEntryManager.Reset();

    for ( auto& driver : g_GpioDriverList )
    {
        driver.SafeCastTo<IGpioDriver>().FinalizeDriver();
    }

    g_InterruptThreadLooper.Finalize();

    g_DriverInitializationState = DriverInitializationState::NotInitialized;
}

void RegisterDriver(IGpioDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(g_DriverInitializationState == DriverInitializationState::NotInitialized,
        "All drivers MUST be registered before nn::gpio::driver::Initialize() is called\n");
    g_GpioDriverList.push_back(*pDriver);
}

void UnregisterDriver(IGpioDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(g_DriverInitializationState == DriverInitializationState::NotInitialized,
        "All drivers MUST be registered before nn::gpio::driver::Initialize() is called\n");
    if ( pDriver->IsLinkedToList() )
    {
        g_GpioDriverList.erase(g_GpioDriverList.iterator_to(*pDriver));
    }
}

nn::Result RegisterDeviceCodeNodeParser(nn::gpio::driver::detail::dt::IDeviceCodeNodeParser* pParser) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pParser);
    NN_SDK_REQUIRES(g_DriverInitializationState != DriverInitializationState::Initialized,
        "Device code node parser MUST be registered before nn::gpio::driver::Initialize() is called or during the driver initialization function IGpioDriver::InitializeDriver()\n");
    g_GpioDeviceCodeNodeRootParser.RegisterParser(pParser);
    NN_RESULT_SUCCESS;
}

nn::Result RegisterDeviceCode(nn::DeviceCode deviceCode, Pad* pPad) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pPad);
    NN_SDK_REQUIRES(g_DriverInitializationState == DriverInitializationState::Initializing,
        "Device code entry MUST be registered during the driver initialization function IGpioDriver::InitializeDriver()\n");
    NN_RESULT_DO(g_GpioDeviceCodeEntryManager.Add(deviceCode, pPad));
    NN_RESULT_SUCCESS;
}

bool UnregisterDeviceCode(nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_DriverInitializationState == DriverInitializationState::Initializing,
        "Device code entry MUST be registered during the driver initialization function IGpioDriver::InitializeDriver()\n");
    return g_GpioDeviceCodeEntryManager.Remove(deviceCode);
}

nn::Result RegisterInterruptHandler(nn::ddsf::IEventHandler* pHandler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pHandler);
    NN_SDK_REQUIRES(g_DriverInitializationState != DriverInitializationState::NotInitialized,
        "Interrupt handler CANNOT be registered before nn::gpio::driver::Initialize() is called\n");
    g_InterruptThreadLooper.RegisterHandler(pHandler);
    NN_RESULT_SUCCESS;
}

void UnregisterInterruptHandler(nn::ddsf::IEventHandler* pHandler) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pHandler);
    NN_SDK_REQUIRES(g_DriverInitializationState != DriverInitializationState::NotInitialized,
        "Interrupt handler CANNOT be unregistered before nn::gpio::driver::Initialize() is called\n");
    g_InterruptThreadLooper.UnregisterHandler(pHandler);
}

IGpioDriver::List& GetDriverList() NN_NOEXCEPT
{
    return g_GpioDriverList;
}

nn::Result FindPad(Pad** ppOutPad, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_DriverInitializationState == DriverInitializationState::Initialized,
        "FindPad MUST be called after nn::gpio::driver::Initialize() is done\n");
    NN_SDK_ASSERT_NOT_NULL(ppOutPad);
    nn::ddsf::IDevice* pDevice;
    NN_RESULT_DO(g_GpioDeviceCodeEntryManager.FindDevice(&pDevice, deviceCode));
    *ppOutPad = pDevice->SafeCastToPointer<Pad>();
    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
// ForDev 系の pad number 直指定 API 向けデバイスコード変換ユーティリティ
nn::Result FindPadByNumber(Pad** ppOutPad, int padNumber) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_DriverInitializationState == DriverInitializationState::Initialized,
        "FindPadByNumber MUST be called after nn::gpio::driver::Initialize() is done\n");
    NN_SDK_ASSERT_NOT_NULL(ppOutPad);
    if ( !ppOutPad )
    {
        return nn::gpio::ResultUnknown();
    }

    bool isFound = false;
    g_GpioDeviceCodeEntryManager.ForEachEntry(
        [&](nn::ddsf::DeviceCodeEntry* pEntry) NN_NOEXCEPT -> bool
        {
            NN_SDK_ASSERT_NOT_NULL(pEntry);
            auto& pad = pEntry->GetDevice().SafeCastTo<Pad>();
            if ( pad.GetPadNumber() == padNumber )
            {
                isFound = true;
                *ppOutPad = &pad;
                return false; // Break ForEach loop
            }
            return true;
        }
    );
    if ( !isFound )
    {
        NN_DETAIL_GPIO_ERROR("Unsupported pad number (0x%x) is specified\n", padNumber);
        return nn::gpio::ResultDeviceNotFound();
    }
    NN_RESULT_SUCCESS;
}

}}}} // nn::gpio::driver::detail
