﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   GPIO ドライバの　Interrupt 周りの管理クラス(UART のものの GPIO 版)
 * @details  GPIO ドライバの　Interrupt 周りの管理クラス(Process が出来たらそちらに移行予定)
 */

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_IntrusiveList.h>

#include <nn/gpio/driver/gpio_PadAccessor.h>

#include "gpio_DdUtil.h"
#include "gpio_RegAccessor-soc.tegra.h"
#include "gpio_EventHolder-soc.tegra.h"

namespace nn {
namespace gpio {
namespace driver {
namespace detail {

/**
 * @brief   割り込みイベント通知用コールバック関数の型宣言
 */
typedef void (*InterruptCallbackFuncType)(uintptr_t arg);

/**
 * @brief   割り込みイベント通知クラス。
 * @details 内部に持っているスタック・優先度でスレッドを立てる。@n
 *          スレッド内で 1 つ以上の割り込みイベントを多重待ちする。@n
 *          割り込みが起こると、初期化時に登録したコールバックに割り込み番号を通知する。@n
 *          シングルトンパターンのため実体は常に唯一である。
 */
class InterruptNotifier
{
    NN_DISALLOW_COPY(InterruptNotifier);
    NN_DISALLOW_MOVE(InterruptNotifier);

    typedef nn::util::IntrusiveList<PadEventHolder, nn::util::IntrusiveListMemberNodeTraits<PadEventHolder, &PadEventHolder::m_Node>>  BoundPadEventHolderList;

private: // Singleton
    InterruptNotifier() NN_NOEXCEPT :
        m_IsInitialized(false),
        m_InterruptEventMutex(false),
        m_FinalizeRequestEvent(nn::os::EventClearMode_ManualClear)
        {
            m_GpioVirtualAddress = nn::dd::QueryIoMappingAddress(GpioPhysicalAddress, GpioAddressSize);
            if (m_GpioVirtualAddress == 0)
            {
                // physicalAddress が指す I/O アドレスがマッピングされていない
                NN_ABORT("I/O registers for 0x%llx are not mapped. Make sure the capability setting is properly set for this process.\n", m_GpioVirtualAddress);
            }
        }

public:
    /**
     * @brief   本クラスの唯一のインスタンスを取得する。
     */
    static InterruptNotifier& GetInstance() NN_NOEXCEPT
    {
        static InterruptNotifier inst;
        return inst;
    }

    /**
     * @brief   初期化。割り込みイベントを設定する。
     *          また、イベント通知スレッドを作成し、スレッドを開始する。
     * @pre     初期化前の状態であること。
     * @post    初期化完了。スレッドが作成され、停止した状態である。
     */
    void Initialize() NN_NOEXCEPT;

    /**
     * @brief   クラスの終了。割り込みイベントとの紐付を破棄し、初期化時に作成したスレッドを停止し、破棄する。
     * @pre     クラスが初期化された状態であること。
     * @post    初期化前の状態に戻る。
     */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief   Session を管理しているリストへ Session を追加します。
    */
    nn::Result AddEventHolderToBoundList(PadEventHolder* pEventHolder) NN_NOEXCEPT;

    /**
    * @brief   Session を管理しているリストから Session を削除します。
    */
    void DeleteEventHolderFromBoundList(PadEventHolder* pEventHolder) NN_NOEXCEPT;

    /**
    * @brief   InterruptEvent 周りを排他する Mutex のポインタを取得します。
    */
    nn::os::Mutex* GetInterruptEventMutexPointer() NN_NOEXCEPT
    {
        return &m_InterruptEventMutex;
    }

private:
    static void InterruptThread(void* arg) NN_NOEXCEPT;
    void InterruptThreadBody() NN_NOEXCEPT;

    /**
     * @brief   イベント通知スレッドの開始。
     * @pre     クラスが初期化された状態であること。スレッドが停止した状態であること。
     * @post    スレッドが開始した状態になる。
     */
    void StartThread() NN_NOEXCEPT;

    /**
     * @brief   イベント通知スレッドの停止。
     * @pre     クラスが初期化された状態であること。スレッドが開始した状態であること。
     * @post    スレッドが停止した状態になる。
     */
    void StopThread() NN_NOEXCEPT;

    // サポートされているパッドの中から割り込みを受けたパッドに紐付けられたパッドを探す
    nn::os::SystemEventType* SearchInterrupedPadEvent(int interruptIdx) NN_NOEXCEPT;

private:

    bool                            m_IsInitialized;
    int                             m_BoundEventCount;
    uintptr_t                       m_GpioVirtualAddress;

    // イベントを受けるスレッド本体と、それに使う多重待ちオブジェクト
    nn::os::ThreadType              m_WorkThread;
    nn::os::MultiWaitType           m_Waiter;

    // 内部の割り込みイベント用(GPIOコントローラー分用意)
    nn::os::MultiWaitHolderType     m_InterruptEventHolder[GpioController_NumOfGpioController];
    InternalInterruptEvent          m_InterruptEvent[GpioController_NumOfGpioController];

    // 紐付済み Session を管理するリスト
    BoundPadEventHolderList         m_BoundPadEventHolderList;

    // 割り込み処理周り全体を排他する mutex
    nn::os::Mutex                   m_InterruptEventMutex;

    // スレッド終了イベント用
    nn::os::MultiWaitHolderType     m_FinalizeRequestEventHolder;
    nn::os::Event                   m_FinalizeRequestEvent;
};

} // detail
} // driver
} // gpio
} // nn
