﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/friends_NotificationQueue.h>
#include <nn/friends/friends_Result.h>
#include <nn/friends/detail/friends_ApiDetail.h>
#include <nn/friends/detail/friends_ShimLibraryGlobal.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>
#include <mutex>

namespace nn { namespace friends {

NotificationQueue::NotificationQueue() NN_NOEXCEPT :
    m_Mutex(true),
    m_Uid(nn::account::InvalidUid),
    m_Handle(nullptr)
{
}

NotificationQueue::~NotificationQueue() NN_NOEXCEPT
{
    Finalize();
}

nn::Result NotificationQueue::Initialize(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckUserStatus(uid));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_SDK_ASSERT(!m_Handle);

    detail::ipc::INotificationService* handle = nullptr;
    NN_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().CreateNotificationService(&handle, uid));

    nn::sf::NativeHandle eventHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(handle->GetEvent(&eventHandle));

    m_Event.AttachReadableHandle(eventHandle.GetOsHandle(), eventHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
    eventHandle.Detach();

    m_Uid = uid;
    m_Handle = handle;

    NN_RESULT_SUCCESS;
}

nn::Result NotificationQueue::Initialize() NN_NOEXCEPT
{
    return Initialize(detail::GetCurrentUser());
}

void NotificationQueue::Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_Uid = nn::account::InvalidUid;

    if (m_Handle)
    {
        nn::sf::ReleaseSharedObject(m_Handle);
        m_Handle = nullptr;

        nn::os::DestroySystemEvent(m_Event.GetBase());
    }
}

const nn::account::Uid& NotificationQueue::GetUid() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return m_Uid;
}

nn::os::SystemEvent* NotificationQueue::GetSystemEvent() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_Handle)
    {
        return &m_Event;
    }
    else
    {
        return nullptr;
    }
}

nn::Result NotificationQueue::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotInitialized());

    NN_RESULT_DO(m_Handle->Clear());

    NN_RESULT_SUCCESS;
}

nn::Result NotificationQueue::Pop(NotificationInfo* outInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInfo);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotInitialized());

    detail::ipc::SizedNotificationInfo info = {};
    NN_RESULT_DO(m_Handle->Pop(&info));

    *outInfo = info.Unbox();

    NN_RESULT_SUCCESS;
}

}}
