﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/friends_Api.h>
#include <nn/friends/friends_Result.h>
#include <nn/friends/detail/friends_ApiDetail.h>
#include <nn/friends/detail/friends_Async.h>
#include <nn/friends/detail/friends_ShimLibraryGlobal.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace friends {

void Initialize() NN_NOEXCEPT
{
    detail::ShimLibraryGlobal::GetInstance().Initialize();
}

void SetAllocator(nn::AllocateFunction allocateFunction, nn::DeallocateFunction freeFunction) NN_NOEXCEPT
{
    detail::SetAsyncAllocator(allocateFunction, freeFunction);
}

nn::Result GetFriendList(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
    const nn::account::Uid& uid, int offset, int count, const FriendFilter& filter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outAccountIds);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    detail::ipc::SizedFriendFilter sizedFilter;
    sizedFilter.Box(filter);

    int32_t actualCount = 0;

    NN_RESULT_DO(session->GetFriendListIds(
        &actualCount,
        nn::sf::OutArray<nn::account::NetworkServiceAccountId>(outAccountIds, static_cast<size_t>(count)),
        uid,
        static_cast<int32_t>(offset),
        sizedFilter, 0));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result GetFriendList(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
    int offset, int count, const FriendFilter& filter) NN_NOEXCEPT
{
    return GetFriendList(outCount, outAccountIds, detail::GetCurrentUser(), offset, count, filter);
}

nn::Result GetFriendList(int* outCount, Friend* outFriends,
    const nn::account::Uid& uid, int offset, int count, const FriendFilter& filter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outFriends);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    detail::ipc::SizedFriendFilter sizedFilter;
    sizedFilter.Box(filter);

    int32_t actualCount = 0;

    NN_RESULT_DO(session->GetFriendList(
        &actualCount,
        nn::sf::OutArray<detail::FriendImpl>(reinterpret_cast<detail::FriendImpl*>(outFriends), static_cast<size_t>(count)),
        uid,
        static_cast<int32_t>(offset),
        sizedFilter, 0));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result GetFriendList(int* outCount, Friend* outFriends,
    int offset, int count, const FriendFilter& filter) NN_NOEXCEPT
{
    return GetFriendList(outCount, outFriends, detail::GetCurrentUser(), offset, count, filter);
}

nn::Result UpdateFriendInfo(Friend* outFriends,
    const nn::account::Uid& uid, const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outFriends);
    NN_SDK_REQUIRES_NOT_NULL(accountIds);
    NN_SDK_REQUIRES(count > 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->UpdateFriendInfo(
        nn::sf::OutArray<detail::FriendImpl>(reinterpret_cast<detail::FriendImpl*>(outFriends), static_cast<size_t>(count)),
        uid,
        nn::sf::InArray<nn::account::NetworkServiceAccountId>(accountIds, static_cast<size_t>(count)), 0));

    NN_RESULT_SUCCESS;
}

nn::Result UpdateFriendInfo(Friend* outFriends,
    const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    return UpdateFriendInfo(outFriends, detail::GetCurrentUser(), accountIds, count);
}

nn::Result GetBlockedUserList(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outAccountIds);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    int32_t actualCount = 0;

    NN_RESULT_DO(session->GetBlockedUserListIds(
        &actualCount,
        nn::sf::OutArray<nn::account::NetworkServiceAccountId>(outAccountIds, static_cast<size_t>(count)),
        uid,
        static_cast<int32_t>(offset)));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result GetBlockedUserList(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
    int offset, int count) NN_NOEXCEPT
{
    return GetBlockedUserList(outCount, outAccountIds, detail::GetCurrentUser(), offset, count);
}

nn::Result GetProfileList(AsyncContext* outAsync, Profile* outProfiles,
    const nn::account::Uid& uid, const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outProfiles);
    NN_SDK_REQUIRES_NOT_NULL(accountIds);
    NN_SDK_REQUIRES_MINMAX(count, 1, GetProfileCountMax);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParamForShallowCopy
    {
        detail::ProfileImpl* outProfiles;
        nn::account::Uid uid;
        const nn::account::NetworkServiceAccountId* accountIds;
        int count;
    }
    ap = {reinterpret_cast<detail::ProfileImpl*>(outProfiles), uid, accountIds, count};

    struct AsyncParam
    {
        detail::ProfileImpl* outProfiles;
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountIds[GetProfileCountMax];
        int count;
    };

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetProfileList",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->GetProfileList(
                nn::sf::OutArray<detail::ProfileImpl>(ap->outProfiles, static_cast<size_t>(ap->count)),
                ap->uid,
                nn::sf::InArray<nn::account::NetworkServiceAccountId>(ap->accountIds, static_cast<size_t>(ap->count))));

            NN_RESULT_SUCCESS;
        },
        [](const void* source, void* dest, size_t size) -> void
        {
            NN_SDK_ASSERT(sizeof (AsyncParam) <= size);
            NN_UNUSED(size);

            const AsyncParamForShallowCopy* sp = static_cast<const AsyncParamForShallowCopy*>(source);
            AsyncParam* dp = static_cast<AsyncParam*>(dest);

            dp->outProfiles = sp->outProfiles;
            dp->uid = sp->uid;
            std::memcpy(dp->accountIds, sp->accountIds, sizeof (nn::account::NetworkServiceAccountId) * sp->count);
            dp->count = sp->count;
        },
        &ap));

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileList(AsyncContext* outAsync, Profile* outProfiles,
    const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    return GetProfileList(outAsync, outProfiles, detail::GetCurrentUser(), accountIds, count);
}

nn::Result DeclareOpenOnlinePlaySession(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->DeclareOpenOnlinePlaySession(uid));

    NN_RESULT_SUCCESS;
}

nn::Result DeclareOpenOnlinePlaySession() NN_NOEXCEPT
{
    return DeclareOpenOnlinePlaySession(detail::GetCurrentUser());
}

nn::Result DeclareCloseOnlinePlaySession(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->DeclareCloseOnlinePlaySession(uid));

    NN_RESULT_SUCCESS;
}

nn::Result DeclareCloseOnlinePlaySession() NN_NOEXCEPT
{
    return DeclareCloseOnlinePlaySession(detail::GetCurrentUser());
}

nn::Result GetPlayHistoryRegistrationKey(PlayHistoryRegistrationKey* outKey,
    const nn::account::Uid& uid, bool isLocalPlay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outKey);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->GetPlayHistoryRegistrationKey(outKey, uid, isLocalPlay));

    NN_RESULT_SUCCESS;
}

nn::Result GetPlayHistoryRegistrationKey(PlayHistoryRegistrationKey* outKey,
    bool isLocalPlay) NN_NOEXCEPT
{
    return GetPlayHistoryRegistrationKey(outKey, detail::GetCurrentUser(), isLocalPlay);
}

nn::Result AddPlayHistory(const nn::account::Uid& uid,
    const PlayHistoryRegistrationKey& key,
    const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    InAppScreenName inAppScreenNameNormalized;
    NN_RESULT_DO(detail::NormalizeInAppScreenName(&inAppScreenNameNormalized, inAppScreenName));

    InAppScreenName myInAppScreenNameNormalized;
    NN_RESULT_DO(detail::NormalizeInAppScreenName(&myInAppScreenNameNormalized, myInAppScreenName));

    NN_RESULT_DO(session->AddPlayHistory(uid, key, inAppScreenNameNormalized, myInAppScreenNameNormalized, 0));

    NN_RESULT_SUCCESS;
}

nn::Result AddPlayHistory(const PlayHistoryRegistrationKey& key,
    const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    return AddPlayHistory(detail::GetCurrentUser(), key, inAppScreenName, myInAppScreenName);
}

}}
