﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/json/friends_JsonHttpInputStream.h>
#include <nn/friends/friends_Result.h>
#include <nn/friends/friends_ResultPrivate.h>
#include <nn/friends/detail/service/util/friends_HttpErrorHandler.h>
#include <cstdarg>
#include <cstdio>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>

#define CR 0x0D // NOLINT(preprocessor/const)
#define LF 0x0A // NOLINT(preprocessor/const)

#define CURL_EASY_DO(exp) \
    do                                                              \
    {                                                               \
        CURLcode code_ = (exp);                                     \
                                                                    \
        if (code_ != CURLE_OK)                                      \
        {                                                           \
            return detail::service::util::HandleHttpError(code_);   \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

JsonHttpInputStream::JsonHttpInputStream() NN_NOEXCEPT :
    m_Curl(nullptr),
    m_Multi(nullptr),
    m_Headers(nullptr),
    m_Param(),
    m_ResponseHeaderCallback(nullptr),
    m_ResponseHeaderParam(nullptr),
    m_ResponseHeaderEndCallback(nullptr),
    m_ResponseHeaderEndParam(nullptr),
    m_StatusCode(0),
    m_IsOpened(false)
{
}

JsonHttpInputStream::~JsonHttpInputStream() NN_NOEXCEPT
{
    Close();
}

nn::Result JsonHttpInputStream::Open(const char* url) NN_NOEXCEPT
{
    return Open("GET", url);
}

nn::Result JsonHttpInputStream::Open(const char* request, const char* url) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(request);
    NN_SDK_REQUIRES_NOT_NULL(url);

    NN_SDK_ASSERT(!m_IsOpened);

    CURL* curl = curl_easy_init();

    if (!curl)
    {
        NN_RESULT_THROW(ResultHttpErrorInterfaceFailed());
    }

    NN_UTIL_SCOPE_EXIT
    {
        if (curl)
        {
            curl_easy_cleanup(curl);
        }
    };

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_CUSTOMREQUEST, request));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_URL, url));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, SslCtxFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, HttpHeaderFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HEADERDATA, this));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, HttpWriteFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEDATA, &m_Param));

    // CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_VERBOSE, 1));

    // 最大リダイレクト回数を制限する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 1));
    // 1 B/s 以下の転送速度の通信が 30 秒以上続いたら切断する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_LIMIT, 1));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_TIME, 30));
    // 接続タイムアウト（CONNECT 完了までの時間）を 30 秒にする。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 30));
    // 証明書の時刻検証を無効化する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0));

    CURLM* multi = curl_multi_init();

    if (!multi)
    {
        NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
    }

    NN_UTIL_SCOPE_EXIT
    {
        if (multi)
        {
            curl_multi_cleanup(multi);
        }
    };

    CURLMcode code = curl_multi_add_handle(multi, curl);

    if (code != CURLM_OK)
    {
        curl_multi_cleanup(multi);

        if (code == CURLM_OUT_OF_MEMORY)
        {
            NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
        }
        else
        {
            NN_RESULT_THROW(ResultHttpErrorFailedInit());
        }
    }

    m_Curl = curl;
    curl = nullptr;

    m_Multi = multi;
    multi = nullptr;

    m_Param.readNext = true;
    m_Param.continuousWrite = false;

    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

void JsonHttpInputStream::Close() NN_NOEXCEPT
{
    if (m_IsOpened)
    {
        curl_multi_remove_handle(m_Multi, m_Curl);
        curl_multi_cleanup(m_Multi);

        if (m_Headers)
        {
            curl_slist_free_all(m_Headers);
        }

        curl_easy_cleanup(m_Curl);

        m_IsOpened = false;
    }
}

nn::Result JsonHttpInputStream::AddRequestHeader(const char* format, ...) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(format);

    NN_SDK_ASSERT(m_IsOpened);

    char line[RequestHeaderLineSize] = {};

    va_list args;
    va_start(args, format);

    int length = nn::util::VSNPrintf(line, sizeof (line), format, args);

    va_end(args);

    if (length >= static_cast<int>(RequestHeaderLineSize))
    {
        NN_RESULT_THROW(ResultInvalidArgument());
    }

    curl_slist* headers = curl_slist_append(m_Headers, line);

    if (!headers)
    {
        NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
    }

    m_Headers = headers;

    CURL_EASY_DO(curl_easy_setopt(m_Curl, CURLOPT_HTTPHEADER, m_Headers));

    NN_RESULT_SUCCESS;
}

nn::Result JsonHttpInputStream::SetPostField(const char* data, bool copy) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(data);

    NN_SDK_ASSERT(m_IsOpened);

    CURL_EASY_DO(curl_easy_setopt(m_Curl, copy ? CURLOPT_COPYPOSTFIELDS : CURLOPT_POSTFIELDS, data));

    NN_RESULT_SUCCESS;
}

void JsonHttpInputStream::RegisterResponseHeaderCallback(ResponseHeaderCallback function, void* param) NN_NOEXCEPT
{
    m_ResponseHeaderCallback = function;
    m_ResponseHeaderParam = param;
}

void JsonHttpInputStream::RegisterResponseHeaderEndCallback(ResponseHeaderEndCallback function, void* param) NN_NOEXCEPT
{
    m_ResponseHeaderEndCallback = function;
    m_ResponseHeaderEndParam = param;
}

nn::Result JsonHttpInputStream::FillImpl(size_t* outRead, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsOpened);

    if (IsCanceled())
    {
        NN_RESULT_THROW(ResultCanceled());
    }

    m_Param.read = 0;

    if (m_Param.readNext)
    {
        m_Param.stream = buffer;
        m_Param.streamSize = bufferSize;
        m_Param.positionInHttpBuffer = 0;
        m_Param.readNext = false;

        if (m_Param.continuousWrite)
        {
            m_Param.continuousWrite = false;
            curl_easy_pause(m_Curl, CURLPAUSE_CONT);
        }
    }
    else
    {
        curl_easy_pause(m_Curl, CURLPAUSE_CONT);
    }

    int stillRunning = 0;

    while (m_Param.read == 0)
    {
        int rval = 0;
        CURLMcode code = curl_multi_wait(m_Multi, nullptr, 0, 10, &rval);

        if (code != CURLM_OK)
        {
            if (code == CURLM_OUT_OF_MEMORY)
            {
                NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
            }
            else
            {
                NN_RESULT_THROW(ResultHttpErrorBadFunctionArgument());
            }
        }

        if (rval == -1)
        {
            NN_RESULT_THROW(ResultHttpErrorRecvError());
        }

        curl_multi_perform(m_Multi, &stillRunning);

        if (stillRunning == 0)
        {
            break;
        }

        if (IsCanceled())
        {
            NN_RESULT_THROW(ResultCanceled());
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    *outRead = m_Param.read;

    if (stillRunning == 0)
    {
        CURLMsg* msg = nullptr;

        do
        {
            int n = 0;
            msg = curl_multi_info_read(m_Multi, &n);

            if (msg && (msg->msg == CURLMSG_DONE))
            {
                return detail::service::util::HandleHttpError(msg->data.result);
            }
        }
        while (msg);
    }

    NN_RESULT_SUCCESS;
}

CURLcode JsonHttpInputStream::SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);

    nn::ssl::Context* context = reinterpret_cast<nn::ssl::Context*>(ssl);

    if (context->Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
    {
        return CURLE_ABORTED_BY_CALLBACK;
    }

    return CURLE_OK;
}

size_t JsonHttpInputStream::HttpHeaderFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    JsonHttpInputStream* stream = static_cast<JsonHttpInputStream*>(param);

    size_t bufferSize = size * count;

    if (bufferSize >= sizeof ("HTTP/X.Y SSS ") - 1 && nn::util::Strncmp(buffer, "HTTP/", 5) == 0)
    {
        int majorVersion = 0;
        int minorVersion = 0;
        int statusCode = 0;

        char space = '\0';

        int result = std::sscanf(buffer, "HTTP/%1d.%1d %03d%c", &majorVersion, &minorVersion, &statusCode, &space);

        if (result == 4 && space == ' ')
        {
            stream->m_StatusCode = statusCode;
        }

        return bufferSize;
    }

    // 1xx 系のステータスコードは内部処理に影響を与えるものであるため、アプリに影響を与えないようにする。
    if (stream->m_StatusCode / 100 == 1)
    {
        return bufferSize;
    }

    if (bufferSize == 2 && buffer[0] == CR && buffer[1] == LF)
    {
        if (stream->m_ResponseHeaderEndCallback)
        {
            bool result = stream->m_ResponseHeaderEndCallback(stream->m_StatusCode, stream->m_ResponseHeaderEndParam);

            if (!result)
            {
                bufferSize = 0;
            }
        }

        return bufferSize;
    }

    char* end = buffer + bufferSize;

    char* label = buffer;
    char* colon = reinterpret_cast<char*>(std::memchr(buffer, ':', bufferSize));

    if (!colon)
    {
        return bufferSize;
    }

    char* value = colon + 1;

    if (value == end)
    {
        return bufferSize;
    }

    while ((*value == ' ' || *value == '\t') && value != end)
    {
        value++;
    }

    size_t remainSize = end - value;

    // 必ず改行コード（CRLF）は存在するはず。
    if (remainSize < 2)
    {
        return bufferSize;
    }

    char* cr = reinterpret_cast<char*>(std::memchr(value, CR, remainSize));

    if (!cr)
    {
        return false;
    }

    *colon = '\0';
    *cr = '\0';

    if (stream->m_ResponseHeaderCallback)
    {
        bool result = stream->m_ResponseHeaderCallback(label, value, stream->m_ResponseHeaderParam);

        if (!result)
        {
            bufferSize = 0;
        }
    }

    *colon = ':';
    *cr = CR;

    return bufferSize;
}

size_t JsonHttpInputStream::HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    HttpWriteFunctionParam* p = reinterpret_cast<HttpWriteFunctionParam*>(param);

    size_t bufferSize = size * count;

    if (p->read != 0)
    {
        NN_SDK_ASSERT(p->readNext);

        // 読み込まれたデータがパースされる前に本コールバックが再度呼び出された場合、次のデータを curl_easy_pause(CONT) で読み込む。
        p->continuousWrite = true;

        return CURL_WRITEFUNC_PAUSE;
    }

    size_t streamSize = bufferSize - p->positionInHttpBuffer;

    size_t copySize = (streamSize < p->streamSize ? streamSize : p->streamSize);

    std::memcpy(p->stream, &buffer[p->positionInHttpBuffer], copySize);
    p->read = copySize;
    p->positionInHttpBuffer += copySize;

    NN_SDK_ASSERT(p->positionInHttpBuffer <= bufferSize);

    if (bufferSize == p->positionInHttpBuffer)
    {
        // WriteFunction のバッファをすべてパースしたら、次のデータを読み込む。
        p->readNext = true;

        return bufferSize;
    }
    else
    {
        return CURL_WRITEFUNC_PAUSE;
    }
}

}}}}}
