﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/friends_NotificationService.h>
#include <nn/friends/detail/service/friends_Common.h>
#include <nn/friends/detail/service/core/friends_NotificationEventHandler.h>

namespace nn { namespace friends { namespace detail { namespace service {

NotificationService::NotificationService(const Capability& capability, const nn::account::Uid& uid) NN_NOEXCEPT :
    m_Mutex(true),
    m_Capability(capability),
    m_IsEventCreated(false),
    m_Uid(uid),
    m_Count(0),
    m_HasFriendListUpdated(false),
    m_HasFriendRequestReceived(false)
{
    detail::service::core::NotificationEventHandler::GetInstance().Attach(this);
}

NotificationService::~NotificationService() NN_NOEXCEPT
{
    detail::service::core::NotificationEventHandler::GetInstance().Detach(this);
}

nn::Result NotificationService::GetEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    if (!m_IsEventCreated)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(m_Event.GetBase(),
            nn::os::EventClearMode_ManualClear, true));

        m_IsEventCreated = true;
    }

    outHandle.Set(nn::sf::NativeHandle(m_Event.DetachReadableHandle(), true));

    NN_RESULT_SUCCESS;
}

nn::Result NotificationService::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_Count = 0;
    m_HasFriendListUpdated = false;
    m_HasFriendRequestReceived = false;

    NN_RESULT_SUCCESS;
}

nn::Result NotificationService::Pop(nn::sf::Out<nn::friends::detail::ipc::SizedNotificationInfo> outInfo) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_Count > 0, ResultNotificationNotFound());

    *outInfo = m_Infos[0];

    m_Count = detail::service::util::ArrayAccessor::RemoveEntry(m_Infos, m_Count, 0);

    if (outInfo->type == detail::ipc::SizedNotificationType_FriendListUpdated)
    {
        m_HasFriendListUpdated = false;
    }
    else if (outInfo->type == detail::ipc::SizedNotificationType_FriendRequestReceived)
    {
        m_HasFriendRequestReceived = false;
    }

    NN_RESULT_SUCCESS;
}

void NotificationService::NotifyFriendListUpdated(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (uid != m_Uid)
    {
        return;
    }

    if (!m_HasFriendListUpdated)
    {
        // フレンドプレゼンスの更新通知をすべて破棄する。
        m_Count = 0;

        m_Infos[m_Count++].type = detail::ipc::SizedNotificationType_FriendListUpdated;
        m_HasFriendListUpdated = true;

        if (m_HasFriendRequestReceived)
        {
            m_Infos[m_Count++].type = detail::ipc::SizedNotificationType_FriendRequestReceived;
        }
    }

    Signal();
}

void NotificationService::NotifyFriendPresenceUpdated(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (uid != m_Uid)
    {
        return;
    }

    bool isExists = m_HasFriendListUpdated;

    if (!isExists)
    {
        for (int i = 0; i < m_Count; i++)
        {
            if (m_Infos[i].type == detail::ipc::SizedNotificationType_FriendPresenceUpdated && m_Infos[i].accountId == accountId)
            {
                isExists = true;
                break;
            }
        }
    }

    if (!isExists)
    {
        if (m_Count == NotificationCountMax)
        {
            // 通知が満杯の時、フレンドプレゼンスの更新通知をフレンドリストの更新通知に丸め込む。
            NotifyFriendListUpdated(uid);
            return;
        }
        else
        {
            NotificationInfo info = {NotificationType_FriendPresenceUpdated, accountId};

            m_Infos[m_Count++].Box(info);
        }
    }

    Signal();
}

void NotificationService::NotifyFriendRequestReceived(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Capability.IsPermitted(Capability::Flag_View))
    {
        return;
    }
    if (uid != m_Uid)
    {
        return;
    }

    if (!m_HasFriendRequestReceived)
    {
        if (m_Count == NotificationCountMax)
        {
            // 通知が満杯の時、フレンドプレゼンスの更新通知をフレンドリストの更新通知に丸め込む。
            NotifyFriendListUpdated(uid);
        }

        NN_SDK_ASSERT(m_Count < NotificationCountMax);

        m_Infos[m_Count++].type = detail::ipc::SizedNotificationType_FriendRequestReceived;
        m_HasFriendRequestReceived = true;
    }

    Signal();
}

void NotificationService::Signal() NN_NOEXCEPT
{
    if (m_IsEventCreated)
    {
        m_Event.Signal();
    }
}

}}}}
