﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskSyncUserSetting.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerUserResource.h>
#include <nn/friends/detail/service/core/friends_UserSettingManager.h>
#include <nn/friends/detail/service/core/friends_PlayLogManager.h>
#include <nn/friends/detail/service/core/friends_WorkBuffer.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

SyncUserSettingTask::SyncUserSettingTask() NN_NOEXCEPT :
    m_IsSetDefaultPlayLogPermissionNeeded(false),
    m_UploadPlayLogCount(0)
{
}

nn::Result SyncUserSettingTask::Main() NN_NOEXCEPT
{
    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepSync());

    bool hasFriendCode;
    NN_RESULT_DO(UserSettingManager::GetInstance().HasFriendCode(&hasFriendCode, GetUid()));

    if (!hasFriendCode)
    {
        NN_RESULT_DO(StepIssueFriendCode());
    }
    if (m_IsSetDefaultPlayLogPermissionNeeded)
    {
        NN_RESULT_DO(StepSetDefaultPlayLogPermission());
    }

    bool hasPlayLog;
    NN_RESULT_DO(PlayLogManager::GetInstance().HasPlayLog(&hasPlayLog, GetUid()));

    if (hasPlayLog)
    {
        NN_RESULT_DO(StepUploadPlayLog());
    }

    NN_RESULT_SUCCESS;
}

nn::Result SyncUserSettingTask::StepSync() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] SyncUserSettingTask::StepSync ...\n");

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users/%016llx", WebApi::GetFqdn(), accountId.id);

    NN_RESULT_DO(stream.Open("GET", url));

    UserResourceHandler handler;
    NN_RESULT_DO(handler.Initialize(WorkBuffer::GetInstance().Get(), WorkBuffer::GetInstance().GetSize()));

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_DO(UserSettingManager::GetInstance().Write(GetUid(), handler.GetResource()));

    if (handler.GetResource().playLogPermission == ExtraPermission_Unconfigured)
    {
        m_IsSetDefaultPlayLogPermissionNeeded = true;
    }

    NN_RESULT_SUCCESS;
}

nn::Result SyncUserSettingTask::StepIssueFriendCode() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] SyncUserSettingTask::StepIssueFriendCode ...\n");

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users/%016llx/generate_code", WebApi::GetFqdn(), accountId.id);

    NN_RESULT_DO(stream.Open("POST", url));
    NN_RESULT_DO(stream.SetPostField("type=NX", false));

    UserResourceHandler handler;
    NN_RESULT_DO(handler.Initialize(WorkBuffer::GetInstance().Get(), WorkBuffer::GetInstance().GetSize()));

    NN_RESULT_TRY(WebApi::Call(handler, stream, GetUid(), GetCancelable()))
        NN_RESULT_CATCH(ResultDisallowFriendCodeIssue)
        {
            // 同タイミングでフレンドコードを発行したかもしれない。
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(UserSettingManager::GetInstance().Write(GetUid(), handler.GetResource()));

    NN_RESULT_SUCCESS;
}

nn::Result SyncUserSettingTask::StepSetDefaultPlayLogPermission() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] SyncUserSettingTask::StepSetDefaultPlayLogPermission ...\n");

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users/%016llx", WebApi::GetFqdn(), accountId.id);

    char patch[512] = {};
    NN_RESULT_DO(CreateSetDefaultPlayLogPermissionPatch(patch, sizeof (patch)));

    NN_RESULT_DO(stream.Open("PATCH", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json-patch+json"));
    NN_RESULT_DO(stream.SetPostField(patch, false));

    UserResourceHandler handler;
    NN_RESULT_DO(handler.Initialize(WorkBuffer::GetInstance().Get(), WorkBuffer::GetInstance().GetSize()));

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_DO(UserSettingManager::GetInstance().Write(GetUid(), handler.GetResource()));

    NN_RESULT_SUCCESS;
}

nn::Result SyncUserSettingTask::CreateSetDefaultPlayLogPermissionPatch(char* buffer, size_t size) NN_NOEXCEPT
{
    nn::friends::ExtraPermission playLogPermission;
    NN_RESULT_DO(UserSettingManager::GetInstance().GetPlayLogPermission(&playLogPermission, GetUid()));

    detail::service::json::FixedSizeAllocator allocator;
    detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    detail::service::json::JsonWriter writer(stream, &allocator, 16);

    detail::service::json::JsonPatchGenerator::BeginWrite(writer);
    {
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/playLog");
        {
            writer.String(playLogPermission == ExtraPermission_Self ? "[]" : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/favoriteFriends/playLog");
        {
            writer.String(playLogPermission == ExtraPermission_FavoriteFriends ? "[]" : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/friends/playLog");
        {
            writer.String(playLogPermission == ExtraPermission_Friends ? "[]" : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/foaf/playLog");
        {
            writer.String(playLogPermission == ExtraPermission_FriendsOfFriend ? "[]" : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/everyone/playLog");
        {
            writer.String(playLogPermission == ExtraPermission_Everyone ? "[]" : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
    }
    detail::service::json::JsonPatchGenerator::EndWrite(writer);

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

nn::Result SyncUserSettingTask::StepUploadPlayLog() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] SyncUserSettingTask::StepUploadPlayLog ...\n");

    NN_RESULT_DO(PlayLogManager::GetInstance().GetPlayLogList(&m_UploadPlayLogCount, m_UploadPlayLogs, GetUid(), PlayLogCountMax));

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users/%016llx", WebApi::GetFqdn(), accountId.id);

    // CreateUploadPlayLogPatch と合わせて約 14KB 消費するので注意。
    char patch[6144] = {};
    NN_RESULT_DO(CreateUploadPlayLogPatch(patch, sizeof (patch)));

    NN_RESULT_DO(stream.Open("PATCH", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json-patch+json"));
    NN_RESULT_DO(stream.SetPostField(patch, false));

    UserResourceHandler handler;
    NN_RESULT_DO(handler.Initialize(WorkBuffer::GetInstance().Get(), WorkBuffer::GetInstance().GetSize()));

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_DO(UserSettingManager::GetInstance().Write(GetUid(), handler.GetResource()));
    NN_RESULT_DO(PlayLogManager::GetInstance().RemoveUploadedPlayLog(GetUid(), m_UploadPlayLogs, m_UploadPlayLogCount));

    NN_RESULT_SUCCESS;
}

nn::Result SyncUserSettingTask::CreateUploadPlayLogPatch(char* buffer, size_t size) NN_NOEXCEPT
{
    PlayLog cachedPlayLogs[PlayLogCountMax];
    int cachedCount;

    NN_RESULT_DO(UserSettingManager::GetInstance().GetPlayLogList(&cachedCount, cachedPlayLogs, GetUid(), PlayLogCountMax));

    PlayLog mergedPlayLogs[PlayLogCountMax];
    int mergedCount;

    ParameterConverter::MergePlayLog(&mergedCount, mergedPlayLogs,
        cachedPlayLogs, cachedCount, m_UploadPlayLogs, m_UploadPlayLogCount, PlayLogCountMax);

    char playLogString[5120] = {};
    ParameterConverter::ConvertPlayLog(playLogString, sizeof (playLogString), mergedPlayLogs, mergedCount);

    ExtraPermission permission;
    NN_RESULT_DO(UserSettingManager::GetInstance().GetPlayLogPermission(&permission, GetUid()));

    detail::service::json::FixedSizeAllocator allocator;
    detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    detail::service::json::JsonWriter writer(stream, &allocator, 16);

    detail::service::json::JsonPatchGenerator::BeginWrite(writer);
    {
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/playLog");
        {
            writer.String(permission == ExtraPermission_Self ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/favoriteFriends/playLog");
        {
            writer.String(permission == ExtraPermission_FavoriteFriends ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/friends/playLog");
        {
            writer.String(permission == ExtraPermission_Friends ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/foaf/playLog");
        {
            writer.String(permission == ExtraPermission_FriendsOfFriend ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/everyone/playLog");
        {
            writer.String(permission == ExtraPermission_Everyone ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
    }
    detail::service::json::JsonPatchGenerator::EndWrite(writer);

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

}}}}}
