﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskSendFriendRequest.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerFriendRequestResource.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

SendFriendRequestTask::SendFriendRequestTask() NN_NOEXCEPT :
    m_IsParameterSet(false),
    m_IsParameterSetWithAppInfo(false),
    m_IsParameterSetWithExternalApplicationCatalogId(false),
    m_IsParameterSetWithNintendoNetworkIdInfo(false),
    m_IsSyncNeeded(false)
{
}

nn::Result SendFriendRequestTask::SetParameter(nn::account::NetworkServiceAccountId accountId, RequestType requestType) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(requestType == RequestType_Faced || requestType == RequestType_FriendCode ||
        requestType == RequestType_Facebook || requestType == RequestType_Twitter,
        ResultInvalidArgument());

    m_AccountId = accountId;
    m_RequestType = requestType;

    m_IsParameterSet = true;

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequestTask::SetParameter(nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const ApplicationInfo& appInfo, const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(requestType == RequestType_Faced || requestType == RequestType_InApp, ResultInvalidArgument());

    m_AccountId = accountId;
    m_RequestType = requestType;

    m_AppInfo = appInfo;
    m_InAppScreenName = inAppScreenName;
    m_MyInAppScreenName = myInAppScreenName;

    m_IsParameterSetWithAppInfo = true;

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequestTask::SetParameter(nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const ExternalApplicationCatalogId& catalogId, const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(requestType == RequestType_FriendCandidate, ResultInvalidArgument());

    m_AccountId = accountId;
    m_RequestType = requestType;

    m_CatalogId = catalogId;
    m_InAppScreenName = inAppScreenName;
    m_MyInAppScreenName = myInAppScreenName;

    m_IsParameterSetWithExternalApplicationCatalogId = true;

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequestTask::SetParameter(nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const MiiName& miiName, const MiiImageUrlParam& miiImageUrlParam, const MiiName& myMiiName, const MiiImageUrlParam& myMiiImageUrlParam) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(requestType == RequestType_3Ds || requestType == RequestType_Nnid, ResultInvalidArgument());

    m_AccountId = accountId;
    m_RequestType = requestType;

    m_MiiName = miiName;
    m_MiiImageUrlParam = miiImageUrlParam;
    m_MyMiiName = myMiiName;
    m_MyMiiImageUrlParam = myMiiImageUrlParam;

    m_IsParameterSetWithNintendoNetworkIdInfo = true;

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequestTask::Main() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsParameterSet || m_IsParameterSetWithAppInfo ||
        m_IsParameterSetWithExternalApplicationCatalogId || m_IsParameterSetWithNintendoNetworkIdInfo);

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepSend());

    if (m_IsSyncNeeded)
    {
        NN_RESULT_DO(StepSyncFriendList());
    }

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequestTask::StepSend() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] SendFriendRequestTask::StepSend ...\n");

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/2.0.0/friend_requests", WebApi::GetFqdn());

    char postField[1024] = {};
    NN_RESULT_DO(CreateSendPostField(postField, sizeof (postField)));

    NN_RESULT_DO(stream.Open("POST", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.SetPostField(postField, false));

    FriendRequestResourceHandler handler;
    handler.Initialize(RequestListType_Sent);

    NN_RESULT_TRY(WebApi::Call(handler, stream, GetUid(), GetCancelable()))
        NN_RESULT_CATCH(ResultServerError400SenderFriendCapacityIsFull)
        {
            NN_RESULT_THROW(ResultMyFriendListLimitReached());
        }
        NN_RESULT_CATCH(ResultServerError400ReceiverFriendCapacityIsFull)
        {
            NN_RESULT_THROW(ResultFriendListLimitReached());
        }
        NN_RESULT_CATCH(ResultServerError403OperationIsNotPermitted)
        {
            NN_RESULT_THROW(ResultFriendRequestReceptionNotAllowed());
        }
        NN_RESULT_CATCH(ResultServerError409ResourceDuplicated)
        {
            NN_RESULT_THROW(ResultAlreadyRequested());
        }
    NN_RESULT_END_TRY;

    if (handler.GetResource().requestStatus == RequestStatus_Authorized)
    {
        m_IsSyncNeeded = true;
    }

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequestTask::CreateSendPostField(char* buffer, size_t size) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountId senderId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&senderId, GetUid()));

    nn::friends::detail::service::json::FixedSizeAllocator allocator;
    nn::friends::detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    nn::friends::detail::service::json::JsonWriter writer(stream, &allocator, 16);

    writer.StartObject();
    {
        writer.Key("channels");
        writer.StartArray();
        {
            writer.String(ParameterConverter::ConvertRequestType(m_RequestType));
        }
        writer.EndArray();

        char idString[17];

        nn::util::SNPrintf(idString, sizeof (idString), "%016llx", senderId.id);
        writer.Key("senderId");
        writer.String(idString);

        nn::util::SNPrintf(idString, sizeof (idString), "%016llx", m_AccountId.id);
        writer.Key("receiverId");
        writer.String(idString);

        writer.Key("extras");
        writer.StartObject();
        {
            if (m_IsParameterSetWithAppInfo)
            {
                WriteExtrasWithAppInfo(writer);
            }
            else if (m_IsParameterSetWithExternalApplicationCatalogId)
            {
                WriteExtrasWithExternalApplicationCatalogId(writer);
            }
            else if (m_IsParameterSetWithNintendoNetworkIdInfo)
            {
                WriteExtrasWithNintendoNetworkIdInfo(writer);
            }
            else
            {
                WriteExtras(writer);
            }
        }
        writer.EndObject();
    }
    writer.EndObject();

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

void SendFriendRequestTask::WriteExtras(detail::service::json::JsonWriter& writer) NN_NOEXCEPT
{
    writer.Key("sender");
    writer.StartObject();
    {
        writer.Key("route:sender");
        writer.String("me");
    }
    writer.EndObject();
}

void SendFriendRequestTask::WriteExtrasWithAppInfo(detail::service::json::JsonWriter& writer) NN_NOEXCEPT
{
    writer.Key("sender");
    writer.StartObject();
    {
        writer.Key("route:sender");
        writer.String("me");

        writer.Key("route:name");
        writer.String(m_InAppScreenName.name);

        writer.Key("route:name:language");
        writer.String(m_InAppScreenName.language.string);
    }
    writer.EndObject();

    writer.Key("senderAndReceiver");
    writer.StartObject();
    {
        char idString[17];

        nn::util::SNPrintf(idString, sizeof (idString), "%016llx", m_AppInfo.appId.value);
        writer.Key("route:appInfo:appId");
        writer.String(idString);

        nn::util::SNPrintf(idString, sizeof (idString), "%016llx", m_AppInfo.presenceGroupId);
        writer.Key("route:appInfo:presenceGroupId");
        writer.String(idString);

        writer.Key("route:name");
        writer.String(m_MyInAppScreenName.name);

        writer.Key("route:name:language");
        writer.String(m_MyInAppScreenName.language.string);
    }
    writer.EndObject();
}


void SendFriendRequestTask::WriteExtrasWithExternalApplicationCatalogId(detail::service::json::JsonWriter& writer) NN_NOEXCEPT
{
    writer.Key("sender");
    writer.StartObject();
    {
        writer.Key("route:sender");
        writer.String("me");

        writer.Key("route:name");
        writer.String(m_InAppScreenName.name);

        writer.Key("route:name:language");
        writer.String(m_InAppScreenName.language.string);
    }
    writer.EndObject();

    writer.Key("senderAndReceiver");
    writer.StartObject();
    {
        char catalogIdString[33];

        nn::util::SNPrintf(catalogIdString, sizeof (catalogIdString), "%016llx%016llx", m_CatalogId.value[0], m_CatalogId.value[1]);
        writer.Key("route:candidate:catalogId");
        writer.String(catalogIdString);

        writer.Key("route:name");
        writer.String(m_MyInAppScreenName.name);

        writer.Key("route:name:language");
        writer.String(m_MyInAppScreenName.language.string);
    }
    writer.EndObject();
}

void SendFriendRequestTask::WriteExtrasWithNintendoNetworkIdInfo(detail::service::json::JsonWriter& writer) NN_NOEXCEPT
{
    writer.Key("sender");
    writer.StartObject();
    {
        writer.Key("route:sender");
        writer.String("me");

        writer.Key("route:nnid:miiName");
        writer.String(m_MiiName.value);

        writer.Key("route:nnid:miiImageUrlParam");
        writer.String(m_MiiImageUrlParam.value);
    }
    writer.EndObject();

    writer.Key("senderAndReceiver");
    writer.StartObject();
    {
        writer.Key("route:nnid:miiName");
        writer.String(m_MyMiiName.value);

        writer.Key("route:nnid:miiImageUrlParam");
        writer.String(m_MyMiiImageUrlParam.value);
    }
    writer.EndObject();
}

}}}}}
