﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskChangePlayLogPermission.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerUserResource.h>
#include <nn/friends/detail/service/core/friends_UserSettingManager.h>
#include <nn/friends/detail/service/core/friends_WorkBuffer.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {


ChangePlayLogPermissionTask::ChangePlayLogPermissionTask() NN_NOEXCEPT :
    m_IsParameterSet(false)
{
}

nn::Result ChangePlayLogPermissionTask::SetParameter(ExtraPermission newValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(newValue == ExtraPermission_Self ||
        newValue == ExtraPermission_FavoriteFriends ||
        newValue == ExtraPermission_Friends ||
        newValue == ExtraPermission_FriendsOfFriend ||
        newValue == ExtraPermission_Everyone,
        ResultInvalidArgument());

    m_NewValue = newValue;

    m_IsParameterSet = true;

    NN_RESULT_SUCCESS;
}

nn::Result ChangePlayLogPermissionTask::Main() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsParameterSet);

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepChange());

    NN_RESULT_SUCCESS;
}

nn::Result ChangePlayLogPermissionTask::StepChange() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] ChangePlayLogPermissionTask::StepChange ...\n");

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users/%016llx", WebApi::GetFqdn(), accountId.id);

    // CreateChangePatch と合わせて約 12KB 消費するので注意。
    char patch[6144] = {};
    NN_RESULT_DO(CreateChangePatch(patch, sizeof (patch)));

    NN_RESULT_DO(stream.Open("PATCH", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json-patch+json"));
    NN_RESULT_DO(stream.SetPostField(patch, false));

    UserResourceHandler handler;
    NN_RESULT_DO(handler.Initialize(WorkBuffer::GetInstance().Get(), WorkBuffer::GetInstance().GetSize()));

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_DO(UserSettingManager::GetInstance().Write(GetUid(), handler.GetResource()));

    NN_RESULT_SUCCESS;
}

nn::Result ChangePlayLogPermissionTask::CreateChangePatch(char* buffer, size_t size) NN_NOEXCEPT
{
    PlayLog playLogs[PlayLogCountMax];
    int count;

    NN_RESULT_DO(UserSettingManager::GetInstance().GetPlayLogList(&count, playLogs, GetUid(), PlayLogCountMax));

    char playLogString[5120] = {};
    ParameterConverter::ConvertPlayLog(playLogString, sizeof (playLogString), playLogs, count);

    detail::service::json::FixedSizeAllocator allocator;
    detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    detail::service::json::JsonWriter writer(stream, &allocator, 16);

    detail::service::json::JsonPatchGenerator::BeginWrite(writer);
    {
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/playLog");
        {
            writer.String(m_NewValue == ExtraPermission_Self ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/favoriteFriends/playLog");
        {
            writer.String(m_NewValue == ExtraPermission_FavoriteFriends ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/friends/playLog");
        {
            writer.String(m_NewValue == ExtraPermission_Friends ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/foaf/playLog");
        {
            writer.String(m_NewValue == ExtraPermission_FriendsOfFriend ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/everyone/playLog");
        {
            writer.String(m_NewValue == ExtraPermission_Everyone ? playLogString : "");
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
    }
    detail::service::json::JsonPatchGenerator::EndWrite(writer);

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

}}}}}
