﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskBlockUser.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

BlockUserTask::BlockUserTask() NN_NOEXCEPT :
    m_IsParameterSet(false),
    m_IsParameterSetWithAppInfo(false)
{
}

nn::Result BlockUserTask::SetParameter(nn::account::NetworkServiceAccountId accountId, BlockReason blockReason) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(blockReason == BlockReason_BadFriend || blockReason == BlockReason_BadFriendRequest,
        ResultInvalidArgument());

    m_AccountId = accountId;
    m_BlockReason = blockReason;

    m_IsParameterSet = true;

    NN_RESULT_SUCCESS;
}

nn::Result BlockUserTask::SetParameter(nn::account::NetworkServiceAccountId accountId, BlockReason blockReason,
        const ApplicationInfo& appInfo, const InAppScreenName& inAppScreenName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(blockReason == BlockReason_InApp, ResultInvalidArgument());

    m_AccountId = accountId;
    m_BlockReason = blockReason;

    m_AppInfo = appInfo;
    m_InAppScreenName = inAppScreenName;

    m_IsParameterSetWithAppInfo = true;

    NN_RESULT_SUCCESS;
}

nn::Result BlockUserTask::Main() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsParameterSet || m_IsParameterSetWithAppInfo);

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepBlock());
    NN_RESULT_DO(StepSyncBlockedUserList());

    // フレンドをブロックしたかもしれない。
    NN_RESULT_DO(StepSyncFriendList());
    // 受信したフレンド申請のユーザーをブロックしたかもしれない。
    NN_RESULT_DO(StepGetReceivedFriendRequestCount());

    NN_RESULT_SUCCESS;
}

nn::Result BlockUserTask::StepBlock() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] BlockUserTask::StepBlock ...\n");

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users/%016llx/blocks", WebApi::GetFqdn(), accountId.id);

    char postField[512] = {};
    NN_RESULT_DO(CreateBlockPostField(postField, sizeof (postField)));

    NN_RESULT_DO(stream.Open("POST", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.SetPostField(postField, false));

    detail::service::json::JsonEventHandler handler;

    NN_RESULT_TRY(WebApi::Call(handler, stream, GetUid(), GetCancelable()))
        NN_RESULT_CATCH(ResultServerError404ResourceIsNotFound)
        {
            NN_RESULT_THROW(ResultOwnNetworkServiceAccountNotExistsInBlock());
        }
        NN_RESULT_CATCH(ResultServerError409ResourceAlreadyExists)
        {
            NN_RESULT_THROW(ResultAlreadyBlocked());
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result BlockUserTask::CreateBlockPostField(char* buffer, size_t size) NN_NOEXCEPT
{
    nn::friends::detail::service::json::FixedSizeAllocator allocator;
    nn::friends::detail::service::json::JsonMemoryOutputStream os;

    os.Open(buffer, size);
    os.PutBegin();

    nn::friends::detail::service::json::JsonWriter writer(os, &allocator, 16);

    writer.StartObject();
    {
        char idString[17];

        nn::util::SNPrintf(idString, sizeof (idString), "%016llx", m_AccountId);
        writer.Key("targetUserId");
        writer.String(idString);

        writer.Key("extras");
        writer.StartObject();
        {
            writer.Key("self");
            writer.StartObject();
            {
                writer.Key("reason");
                writer.String(ParameterConverter::ConvertBlockReason(m_BlockReason));

                if (m_IsParameterSetWithAppInfo)
                {
                    nn::util::SNPrintf(idString, sizeof (idString), "%016llx", m_AppInfo.appId.value);
                    writer.Key("route:appInfo:appId");
                    writer.String(idString);

                    nn::util::SNPrintf(idString, sizeof (idString), "%016llx", m_AppInfo.presenceGroupId);
                    writer.Key("route:appInfo:presenceGroupId");
                    writer.String(idString);

                    writer.Key("route:name");
                    writer.String(m_InAppScreenName.name);

                    writer.Key("route:name:language");
                    writer.String(m_InAppScreenName.language.string);
                }
            }
            writer.EndObject();
        }
        writer.EndObject();
    }
    writer.EndObject();

    os.PutEnd();
    os.Close();

    NN_RESULT_SUCCESS;
}

}}}}}
