﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_BgTaskDownloadImage.h>
#include <nn/friends/detail/service/core/friends_ImageDownloader.h>
#include <nn/friends/detail/service/core/friends_ImageDownloadQueueManager.h>
#include <nn/friends/detail/service/core/friends_ImageManager.h>
#include <nn/friends/detail/service/core/friends_FriendListManager.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_WorkBuffer.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

DownloadImageBgTask::DownloadImageBgTask() NN_NOEXCEPT
{
}

nn::Result DownloadImageBgTask::Main() NN_NOEXCEPT
{
    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepDownload());
    NN_RESULT_DO(ImageDownloadQueueManager::GetInstance().Save());

    NN_RESULT_SUCCESS;
}

nn::Result DownloadImageBgTask::StepDownload() NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountId accountId;

    NN_RESULT_TRY(ImageDownloadQueueManager::GetInstance().Pop(&accountId, GetUid()))
        NN_RESULT_CATCH(ResultNetworkServiceAccountNotExistsGeneral)
        {
            NN_RESULT_DO(ImageDownloadQueueManager::GetInstance().Remove(GetUid(), accountId));
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    nn::Result result = StepDownloadImpl(accountId);

    if (result.IsSuccess() || ResultNetworkServiceAccountNotExistsGeneral::Includes(result))
    {
        NN_RESULT_DO(ImageDownloadQueueManager::GetInstance().Remove(GetUid(), accountId));
    }

    NN_RESULT_TRY(ImageDownloadQueueManager::GetInstance().Pop(&accountId, GetUid()))
        NN_RESULT_CATCH(ResultNetworkServiceAccountNotExistsGeneral)
        {
            NN_DETAIL_FRIENDS_INFO("[friends] All profile images are downloaded!\n");
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    BackgroundTaskManager::GetInstance().RequestUpdateFriendProfileImageCache(GetUid());

    NN_RESULT_SUCCESS;
}

nn::Result DownloadImageBgTask::StepDownloadImpl(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] DownloadImageBgTask::StepDownload(%016llx) ...\n", accountId.id);

    FriendDetailedInfoImpl info;

    NN_RESULT_TRY(FriendListManager::GetInstance().GetFriendDetailedInfo(&info, GetUid(), accountId))
        NN_RESULT_CATCH(ResultNotFriend)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    Url& profileImageUrl = info.data.profileExtra.data.profileImageUrl;

    if (!profileImageUrl.value[0])
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    NN_DETAIL_FRIENDS_INFO("[friends] Image url = %s\n", profileImageUrl.value);

    ImageDownloader downloader;

    downloader.SetCancelable(&GetCancelable());

    NN_RESULT_DO(downloader.Perform(profileImageUrl.value, WorkBuffer::GetInstance().Get(), ProfileImageSizeMax));

    // ダウンロード中にフレンドが削除されていないかどうかを確認する。
    NN_RESULT_TRY(FriendListManager::GetInstance().GetFriendDetailedInfo(&info, GetUid(), accountId))
        NN_RESULT_CATCH(ResultNotFriend)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(ImageManager::GetInstance().SaveImage(accountId, profileImageUrl,
        WorkBuffer::GetInstance().Get(), downloader.GetDownloadedSize()));

    NN_RESULT_SUCCESS;
}

}}}}}
