﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerUserResource.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerPlayLog.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

UserResourceHandler::UserResourceHandler() NN_NOEXCEPT :
    m_Resource(nullptr),
    m_Bits()
{
}

nn::Result UserResourceHandler::Initialize(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size >= sizeof (UserResource));

    NN_RESULT_THROW_UNLESS(size >= sizeof (UserResource), ResultInvalidArgument());

    m_Resource = reinterpret_cast<UserResource*>(buffer);

    std::memset(m_Resource, 0, sizeof (UserResource));

    NN_RESULT_SUCCESS;
}

const UserResource& UserResourceHandler::GetResource() const NN_NOEXCEPT
{
    return *m_Resource;
}

bool UserResourceHandler::OnBoolean(const detail::service::json::JsonPath& jsonPath, bool value) NN_NOEXCEPT
{
    if (!m_Bits.friendRequestReception &&
        jsonPath.Compare("$.permissions.friendRequestReception"))
    {
        m_Resource->friendRequestReception = value;
        m_Bits.friendRequestReception = true;
    }
    // TODO: リソース初期化済みフラグ
    else if (jsonPath.Compare("$.extras.self.resourceInitialized"))
    {
        // m_Resource->resourceInitialized = value;
    }

    return true;
}

bool UserResourceHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.friendCodeNextIssuableTime &&
        jsonPath.Compare("$.links.friendCode.regenerableAt"))
    {
        m_Resource->friendCodeNextIssuableTime.value = value.s;
        m_Bits.friendCodeNextIssuableTime = true;
    }

    return true;
}

bool UserResourceHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    if (!m_Bits.accountId &&
        jsonPath.Compare("$.id"))
    {
        nn::account::NetworkServiceAccountId accountId;

        if (detail::util::HexToNumber(&accountId.id, value))
        {
            m_Resource->profileExtra.data.accountId = accountId;
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.nickname &&
        jsonPath.Compare("$.nickname"))
    {
        nn::util::Strlcpy(m_Resource->profileExtra.data.nickname.name, value, sizeof (m_Resource->profileExtra.data.nickname.name));
        m_Bits.nickname = true;
    }
    else if (!m_Bits.profileImageUrl &&
        jsonPath.Compare("$.thumbnailUrl"))
    {
        nn::util::Strlcpy(m_Resource->profileExtra.data.profileImageUrl.value, value,
            sizeof (m_Resource->profileExtra.data.profileImageUrl.value));
        m_Bits.profileImageUrl = true;
    }
    else if (!m_Bits.presencePermission &&
        jsonPath.Compare("$.permissions.presence"))
    {
        m_Resource->presencePermission = ParameterConverter::ConvertPresencePermission(value);
        m_Bits.presencePermission = true;
    }
    else if (!m_Bits.friendCode &&
        jsonPath.Compare("$.links.friendCode.id"))
    {
        nn::util::Strlcpy(m_Resource->friendCode.value, value, sizeof (m_Resource->friendCode.value));
        m_Bits.friendCode = true;
    }
    else if (!m_Bits.playLog &&
        jsonPath.Match("$.extras.*.playLog"))
    {
        if (isOverflowed)
        {
            NN_DETAIL_FRIENDS_WARN("[friends] PlayLog string is overflowed.\n");
        }
        else if (ParsePlayLog(value, length))
        {
            if (jsonPath.Compare("$.extras.self.playLog"))
            {
                m_Resource->playLogPermission = ExtraPermission_Self;
            }
            else if (jsonPath.Compare("$.extras.favoriteFriends.playLog"))
            {
                m_Resource->playLogPermission = ExtraPermission_FavoriteFriends;
            }
            else if (jsonPath.Compare("$.extras.friends.playLog"))
            {
                m_Resource->playLogPermission = ExtraPermission_Friends;
            }
            else if (jsonPath.Compare("$.extras.foaf.playLog"))
            {
                m_Resource->playLogPermission = ExtraPermission_FriendsOfFriend;
            }
            else if (jsonPath.Compare("$.extras.everyone.playLog"))
            {
                m_Resource->playLogPermission = ExtraPermission_Everyone;
            }

            m_Bits.playLog = true;
        }
    }

    return true;
}

bool UserResourceHandler::ParsePlayLog(const char* json, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return false;
    }
    if (json[0] != '[' || json[length - 1] != ']')
    {
        return false;
    }

    PlayLogHandler handler;
    handler.Initialize(m_Resource->profileExtra.data.playLog, PlayLogCountMax);

    detail::service::json::JsonMemoryInputStream stream;
    detail::service::util::Cancelable cancelable;

    char stringBuffer[32];
    char streamBuffer[32];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(json, length);

    return detail::service::json::Parse(handler, stream, cancelable).IsSuccess();
}

}}}}}
