﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerFriendResourceList.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerPlayLog.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerPresence.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

FriendResourceListHandler::FriendResourceListHandler() NN_NOEXCEPT :
    m_Resources(nullptr),
    m_Count(0)
{
}

nn::Result FriendResourceListHandler::Initialize(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size >= sizeof (FriendResource) * FriendCountMax);

    NN_RESULT_THROW_UNLESS(size >= sizeof (FriendResource) * FriendCountMax, ResultInvalidArgument());

    m_Resources = reinterpret_cast<FriendResource*>(buffer);
    m_Count = 0;

    NN_RESULT_SUCCESS;
}

const FriendResource* FriendResourceListHandler::GetResourceList() const NN_NOEXCEPT
{
    return m_Resources;
}

int FriendResourceListHandler::GetResourceCount() const NN_NOEXCEPT
{
    return m_Count;
}

bool FriendResourceListHandler::OnStartObject(const detail::service::json::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.items[*]"))
    {
        if (m_Count < FriendCountMax)
        {
            std::memset(&m_Resources[m_Count], 0, sizeof (FriendResource));
            std::memset(&m_Bits, 0, sizeof (m_Bits));
        }
    }

    return true;
}

bool FriendResourceListHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (jsonPath.Match("$.items[*]"))
    {
        if (m_Bits.accountId && m_Bits.nickname && m_Bits.profileImageUrl)
        {
            if (!m_Bits.isConfirmed)
            {
                m_Resources[m_Count].isNewly = true;
            }
            // オフライン状態の時はログアウト時刻を最終更新時刻扱いにする。
            if (m_Bits.presenceLogoutTime && m_Resources[m_Count].presence.data.status == PresenceStatus_Offline)
            {
                m_Resources[m_Count].presence.data.lastUpdateTime = m_Resources[m_Count].logoutTime;
            }

            if (m_Count < FriendCountMax)
            {
                m_Count++;
            }
        }
    }

    return true;
}

bool FriendResourceListHandler::OnKey(const detail::service::json::JsonPath& jsonPath, const char* key) NN_NOEXCEPT
{
    if (!m_Bits.requestType &&
        jsonPath.Match("$.items[*].channels.*")) // 最初に出現したチャンネルだけで十分。
    {
        m_Resources[m_Count].requestType = ParameterConverter::ConvertRequestType(key);
        m_Bits.requestType = true;
    }

    return true;
}

bool FriendResourceListHandler::OnBoolean(const detail::service::json::JsonPath& jsonPath, bool value) NN_NOEXCEPT
{
    if (!m_Bits.isFavorite &&
        jsonPath.Match("$.items[*].isFavorite"))
    {
        m_Resources[m_Count].isFavorite = value;
        m_Bits.isFavorite = true;
    }
    else  if (!m_Bits.isConfirmed &&
        jsonPath.Match("$.items[*].extras.self.isConfirmed"))
    {
        // 新規＝未確認なので、反転させる。
        m_Resources[m_Count].isNewly = !value;
        m_Bits.isConfirmed = true;
    }
    else  if (!m_Bits.isOnlineNotification &&
        jsonPath.Match("$.items[*].extras.self.isOnlineNotification"))
    {
        m_Resources[m_Count].isOnlineNotification = value;
        m_Bits.isOnlineNotification = true;
    }

    return true;
}

bool FriendResourceListHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.requestRouteTime &&
        jsonPath.Match("$.items[*].createdAt"))
    {
        m_Resources[m_Count].requestRouteInfo.time.value = value.s;
        m_Bits.requestRouteTime = true;
    }
    else if (!m_Bits.presenceLastUpdateTime &&
        jsonPath.Match("$.items[*].friend.presence.updatedAt"))
    {
        m_Resources[m_Count].presence.data.lastUpdateTime.value = value.s;
        m_Bits.presenceLastUpdateTime = true;
    }
    else if (!m_Bits.presenceLogoutTime &&
        jsonPath.Match("$.items[*].friend.presence.logoutAt"))
    {
        m_Resources[m_Count].logoutTime.value = value.s;
        m_Bits.presenceLogoutTime = true;
    }

    return true;
}

bool FriendResourceListHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    if (!m_Bits.accountId &&
        jsonPath.Match("$.items[*].friendId"))
    {
        nn::account::NetworkServiceAccountId accountId;

        if (detail::util::HexToNumber(&accountId.id, value))
        {
            m_Resources[m_Count].accountId = accountId;
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.nickname &&
        jsonPath.Match("$.items[*].friend.nickname"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].nickname.name, value, sizeof (m_Resources[m_Count].nickname.name));
        m_Bits.nickname = true;
    }
    else if (!m_Bits.profileImageUrl &&
        jsonPath.Match("$.items[*].friend.thumbnailUrl"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].profileImageUrl.value, value,
            sizeof (m_Resources[m_Count].profileImageUrl.value));
        m_Bits.profileImageUrl = true;
    }
    else if (!m_Bits.presenceStatus &&
        jsonPath.Match("$.items[*].friend.presence.state"))
    {
        m_Resources[m_Count].presence.data.status = ParameterConverter::ConvertPresenceStatus(value);
        m_Bits.presenceStatus = true;
    }
    else if (!m_Bits.presenceAppId &&
        jsonPath.Match("$.items[*].friend.presence.extras.friends.appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Resources[m_Count].presence.data.lastPlayedApp.appId = appId;
            m_Bits.presenceAppId = true;
        }
    }
    else if (!m_Bits.presencePresenceGroupId &&
        jsonPath.Match("$.items[*].friend.presence.extras.friends.appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Resources[m_Count].presence.data.lastPlayedApp.presenceGroupId = presenceGroupId;
            m_Bits.presencePresenceGroupId = true;
        }
    }
    else if (!m_Bits.presenceAppField &&
        jsonPath.Match("$.items[*].friend.presence.extras.friends.appField"))
    {
        if (isOverflowed)
        {
            NN_DETAIL_FRIENDS_WARN("[friends] The presence string is overflowed.\n");
        }

        m_Bits.presenceAppField = ParseFriendPresenceAppField(value, length);
    }
    else if (!m_Bits.playLog &&
        jsonPath.Match("$.items[*].friend.extras.*.playLog"))
    {
        if (isOverflowed)
        {
            NN_DETAIL_FRIENDS_WARN("[friends] The play log string is overflowed.\n");
        }
        else if (ParsePlayLog(value, length))
        {
            m_Bits.playLog = true;
        }
    }
    else if (!m_Bits.lastPlayRecordAppId &&
        jsonPath.Match("$.items[*].extras.self.lastPlayRecord:appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Resources[m_Count].lastPlayRecord.appInfo.appId = appId;
            m_Bits.lastPlayRecordAppId = true;
        }
    }
    else if (!m_Bits.lastPlayRecordPresenceGroupId &&
        jsonPath.Match("$.items[*].extras.self.lastPlayRecord:appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Resources[m_Count].lastPlayRecord.appInfo.presenceGroupId = presenceGroupId;
            m_Bits.lastPlayRecordPresenceGroupId = true;
        }
    }
    else if (!m_Bits.lastPlayRecordName &&
        jsonPath.Match("$.items[*].extras.self.lastPlayRecord:name"))
    {
        nn::util::CopyUtf8String(m_Resources[m_Count].lastPlayRecord.name.name,
            sizeof (m_Resources[m_Count].lastPlayRecord.name.name), value, length, InAppScreenNameLengthMax);
        m_Bits.lastPlayRecordName = true;
    }
    else if (!m_Bits.lastPlayRecordNameLanguage &&
        jsonPath.Match("$.items[*].extras.self.lastPlayRecord:name:language"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].lastPlayRecord.name.language.string, value,
            sizeof (m_Resources[m_Count].lastPlayRecord.name.language.string));
        m_Bits.lastPlayRecordNameLanguage = true;
    }
    else if (!m_Bits.lastPlayRecordMyName &&
        jsonPath.Match("$.items[*].extras.self.lastPlayRecord:myName"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].lastPlayRecord.myName.name, value,
            sizeof (m_Resources[m_Count].lastPlayRecord.myName.name));
        m_Bits.lastPlayRecordMyName = true;
    }
    else if (!m_Bits.lastPlayRecordMyNameLanguage &&
        jsonPath.Match("$.items[*].extras.self.lastPlayRecord:myName:language"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].lastPlayRecord.myName.language.string, value,
            sizeof (m_Resources[m_Count].lastPlayRecord.myName.language.string));
        m_Bits.lastPlayRecordMyNameLanguage = true;
    }
    else if (!m_Bits.requestRouteAppId &&
        jsonPath.Match("$.items[*].extras.self.route:appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Resources[m_Count].requestRouteInfo.appInfo.appId = appId;
            m_Bits.requestRouteAppId = true;
        }
    }
    else if (!m_Bits.requestRoutePresenceGroupId &&
        jsonPath.Match("$.items[*].extras.self.route:appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Resources[m_Count].requestRouteInfo.appInfo.presenceGroupId = presenceGroupId;
            m_Bits.requestRoutePresenceGroupId = true;
        }
    }
    else if (!m_Bits.requestRouteCatalogId &&
        jsonPath.Match("$.items[*].extras.self.route:candidate:catalogId"))
    {
        if (ParameterConverter::ConvertExternalApplicationCatalogId(&m_Resources[m_Count].requestRouteInfoExtra.candidate.catalogId, value))
        {
            m_Bits.requestRouteCatalogId = true;
        }
    }
    else if (!m_Bits.requestRouteMiiName &&
        jsonPath.Match("$.items[*].extras.self.route:nnid:miiName"))
    {
        nn::util::CopyUtf8String(m_Resources[m_Count].requestRouteInfoExtra.nnid.miiName.value,
            sizeof (m_Resources[m_Count].requestRouteInfoExtra.nnid.miiName.value), value, length, MiiNameLengthMax);
        m_Bits.requestRouteMiiName = true;
    }
    else if (!m_Bits.requestRouteMiiImageUrlParam &&
        jsonPath.Match("$.items[*].extras.self.route:nnid:miiImageUrlParam"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].requestRouteInfoExtra.nnid.miiImageUrlParam.value, value,
            sizeof (m_Resources[m_Count].requestRouteInfoExtra.nnid.miiImageUrlParam.value));
        m_Bits.requestRouteMiiImageUrlParam = true;
    }
    else if (!m_Bits.requestRouteName &&
        jsonPath.Match("$.items[*].extras.self.route:name"))
    {
        nn::util::CopyUtf8String(m_Resources[m_Count].requestRouteInfo.name.name,
            sizeof (m_Resources[m_Count].requestRouteInfo.name.name), value, length, InAppScreenNameLengthMax);
        m_Bits.requestRouteName = true;
    }
    else if (!m_Bits.requestRouteNameLanguage &&
        jsonPath.Match("$.items[*].extras.self.route:name:language"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].requestRouteInfo.name.language.string, value,
            sizeof (m_Resources[m_Count].requestRouteInfo.name.language.string));
        m_Bits.requestRouteNameLanguage = true;
    }

    return true;
} // NOLINT(impl/function_size)

bool FriendResourceListHandler::ParsePlayLog(const char* json, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return false;
    }
    if (json[0] != '[' || json[length - 1] != ']')
    {
        return false;
    }

    PlayLogHandler handler;
    handler.Initialize(m_Resources[m_Count].playLog, PlayLogCountMax);

    detail::service::json::JsonMemoryInputStream stream;
    detail::service::util::Cancelable cancelable;

    char stringBuffer[32];
    char streamBuffer[32];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(json, length);

    return detail::service::json::Parse(handler, stream, cancelable).IsSuccess();
}

bool FriendResourceListHandler::ParseFriendPresenceAppField(const char* json, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return false;
    }
    if (json[0] != '{' || json[length - 1] != '}')
    {
        return false;
    }

    PresenceHandler handler;
    handler.Initialize(m_Resources[m_Count].presence.data.appField, PresenceAppFieldSize);

    detail::service::json::JsonMemoryInputStream stream;
    detail::service::util::Cancelable cancelable;

    char stringBuffer[PresenceAppFieldSize];
    char streamBuffer[32];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(json, length);

    return detail::service::json::Parse(handler, stream, cancelable).IsSuccess();
}

}}}}}
