﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerFriendResource.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerPlayLog.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerPresence.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

FriendResourceHandler::FriendResourceHandler() NN_NOEXCEPT :
    m_Resource(),
    m_Bits()
{
}

const FriendResource& FriendResourceHandler::GetResource() const NN_NOEXCEPT
{
    return m_Resource;
}

bool FriendResourceHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (jsonPath.Compare("$"))
    {
        if (m_Bits.accountId && m_Bits.nickname && m_Bits.profileImageUrl)
        {
            if (!m_Bits.isConfirmed)
            {
                m_Resource.isNewly = true;
            }
            // オフライン状態の時はログアウト時刻を最終更新時刻扱いにする。
            if (m_Bits.presenceLogoutTime && m_Resource.presence.data.status == PresenceStatus_Offline)
            {
                m_Resource.presence.data.lastUpdateTime = m_Resource.logoutTime;
            }
        }
        else
        {
            std::memset(&m_Resource, 0, sizeof (m_Resource));
        }
    }

    return true;
}

bool FriendResourceHandler::OnKey(const detail::service::json::JsonPath& jsonPath, const char* key) NN_NOEXCEPT
{
    if (!m_Bits.requestType &&
        jsonPath.Match("$.channels.*")) // 最初に出現したチャンネルだけで十分。
    {
        m_Resource.requestType = ParameterConverter::ConvertRequestType(key);
        m_Bits.requestType = true;
    }

    return true;
}

bool FriendResourceHandler::OnBoolean(const detail::service::json::JsonPath& jsonPath, bool value) NN_NOEXCEPT
{
    if (!m_Bits.isFavorite &&
        jsonPath.Compare("$.isFavorite"))
    {
        m_Resource.isFavorite = value;
        m_Bits.isFavorite = true;
    }
    else  if (!m_Bits.isConfirmed &&
        jsonPath.Compare("$.extras.self.isConfirmed"))
    {
        // 新規＝未確認なので、反転させる。
        m_Resource.isNewly = !value;
        m_Bits.isConfirmed = true;
    }
    else  if (!m_Bits.isOnlineNotification &&
        jsonPath.Compare("$.extras.self.isOnlineNotification"))
    {
        m_Resource.isOnlineNotification = value;
        m_Bits.isOnlineNotification = true;
    }

    return true;
}

bool FriendResourceHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.requestRouteTime &&
        jsonPath.Compare("$.createdAt"))
    {
        m_Resource.requestRouteInfo.time.value = value.s;
        m_Bits.requestRouteTime = true;
    }
    else if (!m_Bits.presenceLastUpdateTime &&
        jsonPath.Compare("$.friend.presence.updatedAt"))
    {
        m_Resource.presence.data.lastUpdateTime.value = value.s;
        m_Bits.presenceLastUpdateTime = true;
    }
    else if (!m_Bits.presenceLogoutTime &&
        jsonPath.Compare("$.friend.presence.logoutAt"))
    {
        m_Resource.logoutTime.value = value.s;
        m_Bits.presenceLogoutTime = true;
    }

    return true;
}

bool FriendResourceHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    if (!m_Bits.accountId &&
        jsonPath.Compare("$.friendId"))
    {
        nn::account::NetworkServiceAccountId accountId;

        if (detail::util::HexToNumber(&accountId.id, value))
        {
            m_Resource.accountId = accountId;
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.nickname &&
        jsonPath.Compare("$.friend.nickname"))
    {
        nn::util::Strlcpy(m_Resource.nickname.name, value, sizeof (m_Resource.nickname.name));
        m_Bits.nickname = true;
    }
    else if (!m_Bits.profileImageUrl &&
        jsonPath.Compare("$.friend.thumbnailUrl"))
    {
        nn::util::Strlcpy(m_Resource.profileImageUrl.value, value,
            sizeof (m_Resource.profileImageUrl.value));
        m_Bits.profileImageUrl = true;
    }
    else if (!m_Bits.presenceStatus &&
        jsonPath.Compare("$.friend.presence.state"))
    {
        m_Resource.presence.data.status = ParameterConverter::ConvertPresenceStatus(value);
        m_Bits.presenceStatus = true;
    }
    else if (!m_Bits.presenceAppId &&
        jsonPath.Compare("$.friend.presence.extras.friends.appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Resource.presence.data.lastPlayedApp.appId = appId;
            m_Bits.presenceAppId = true;
        }
    }
    else if (!m_Bits.presencePresenceGroupId &&
        jsonPath.Compare("$.friend.presence.extras.friends.appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Resource.presence.data.lastPlayedApp.presenceGroupId = presenceGroupId;
            m_Bits.presencePresenceGroupId = true;
        }
    }
    else if (!m_Bits.presenceAppField &&
        jsonPath.Compare("$.friend.presence.extras.friends.appField"))
    {
        if (isOverflowed)
        {
            NN_DETAIL_FRIENDS_WARN("[friends] The presence string is overflowed.\n");
        }

        m_Bits.presenceAppField = ParseFriendPresenceAppField(value, length);
    }
    else if (!m_Bits.playLog &&
        jsonPath.Match("$.friend.extras.*.playLog"))
    {
        if (isOverflowed)
        {
            NN_DETAIL_FRIENDS_WARN("[friends] The play log string is overflowed.\n");
        }
        else if (ParsePlayLog(value, length))
        {
            m_Bits.playLog = true;
        }
    }
    else if (!m_Bits.lastPlayRecordAppId &&
        jsonPath.Compare("$.extras.self.lastPlayRecord:appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Resource.lastPlayRecord.appInfo.appId = appId;
            m_Bits.lastPlayRecordAppId = true;
        }
    }
    else if (!m_Bits.lastPlayRecordPresenceGroupId &&
        jsonPath.Compare("$.extras.self.lastPlayRecord:appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Resource.lastPlayRecord.appInfo.presenceGroupId = presenceGroupId;
            m_Bits.lastPlayRecordPresenceGroupId = true;
        }
    }
    else if (!m_Bits.lastPlayRecordName &&
        jsonPath.Compare("$.extras.self.lastPlayRecord:name"))
    {
        nn::util::CopyUtf8String(m_Resource.lastPlayRecord.name.name,
            sizeof (m_Resource.lastPlayRecord.name.name), value, length, InAppScreenNameLengthMax);
        m_Bits.lastPlayRecordName = true;
    }
    else if (!m_Bits.lastPlayRecordNameLanguage &&
        jsonPath.Compare("$.extras.self.lastPlayRecord:name:language"))
    {
        nn::util::Strlcpy(m_Resource.lastPlayRecord.name.language.string, value,
            sizeof (m_Resource.lastPlayRecord.name.language.string));
        m_Bits.lastPlayRecordNameLanguage = true;
    }
    else if (!m_Bits.lastPlayRecordMyName &&
        jsonPath.Compare("$.extras.self.lastPlayRecord:myName"))
    {
        nn::util::Strlcpy(m_Resource.lastPlayRecord.myName.name, value, sizeof (m_Resource.lastPlayRecord.myName.name));
        m_Bits.lastPlayRecordMyName = true;
    }
    else if (!m_Bits.lastPlayRecordMyNameLanguage &&
        jsonPath.Compare("$.extras.self.lastPlayRecord:myName:language"))
    {
        nn::util::Strlcpy(m_Resource.lastPlayRecord.myName.language.string, value,
            sizeof (m_Resource.lastPlayRecord.myName.language.string));
        m_Bits.lastPlayRecordMyNameLanguage = true;
    }
    else if (!m_Bits.requestRouteAppId &&
        jsonPath.Compare("$.extras.self.route:appInfo:appId"))
    {
        if (detail::util::HexToNumber(&m_Resource.requestRouteInfo.appInfo.appId.value, value))
        {
            m_Bits.requestRouteAppId = true;
        }
    }
    else if (!m_Bits.requestRoutePresenceGroupId &&
        jsonPath.Compare("$.extras.self.route:appInfo:presenceGroupId"))
    {
        if (detail::util::HexToNumber(&m_Resource.requestRouteInfo.appInfo.presenceGroupId, value))
        {
            m_Bits.requestRoutePresenceGroupId = true;
        }
    }
    else if (!m_Bits.requestRouteCatalogId &&
        jsonPath.Compare("$.extras.self.route:candidate:catalogId"))
    {
        if (ParameterConverter::ConvertExternalApplicationCatalogId(&m_Resource.requestRouteInfoExtra.candidate.catalogId, value))
        {
            m_Bits.requestRouteCatalogId = true;
        }
    }
    else if (!m_Bits.requestRouteMiiName &&
        jsonPath.Compare("$.extras.self.route:nnid:miiName"))
    {
        nn::util::CopyUtf8String(m_Resource.requestRouteInfoExtra.nnid.miiName.value,
            sizeof (m_Resource.requestRouteInfoExtra.nnid.miiName.value), value, length, MiiNameLengthMax);
        m_Bits.requestRouteMiiName = true;
    }
    else if (!m_Bits.requestRouteMiiImageUrlParam &&
        jsonPath.Compare("$.extras.self.route:nnid:miiImageUrlParam"))
    {
        nn::util::Strlcpy(m_Resource.requestRouteInfoExtra.nnid.miiImageUrlParam.value, value,
            sizeof (m_Resource.requestRouteInfoExtra.nnid.miiImageUrlParam.value));
        m_Bits.requestRouteMiiImageUrlParam = true;
    }
    else if (!m_Bits.requestRouteName &&
        jsonPath.Compare("$.extras.self.route:name"))
    {
        nn::util::CopyUtf8String(m_Resource.requestRouteInfo.name.name,
            sizeof (m_Resource.requestRouteInfo.name.name), value, length, InAppScreenNameLengthMax);
        m_Bits.requestRouteName = true;
    }
    else if (!m_Bits.requestRouteNameLanguage &&
        jsonPath.Compare("$.extras.self.route:name:language"))
    {
        nn::util::Strlcpy(m_Resource.requestRouteInfo.name.language.string, value,
            sizeof (m_Resource.requestRouteInfo.name.language.string));
        m_Bits.requestRouteNameLanguage = true;
    }

    return true;
} // NOLINT(impl/function_size)

bool FriendResourceHandler::ParsePlayLog(const char* json, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return false;
    }
    if (json[0] != '[' || json[length - 1] != ']')
    {
        return false;
    }

    PlayLogHandler handler;
    handler.Initialize(m_Resource.playLog, PlayLogCountMax);

    detail::service::json::JsonMemoryInputStream stream;
    detail::service::util::Cancelable cancelable;

    char stringBuffer[32];
    char streamBuffer[32];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(json, length);

    return detail::service::json::Parse(handler, stream, cancelable).IsSuccess();
}

bool FriendResourceHandler::ParseFriendPresenceAppField(const char* json, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return false;
    }
    if (json[0] != '{' || json[length - 1] != '}')
    {
        return false;
    }

    PresenceHandler handler;
    handler.Initialize(m_Resource.presence.data.appField, PresenceAppFieldSize);

    detail::service::json::JsonMemoryInputStream stream;
    detail::service::util::Cancelable cancelable;

    char stringBuffer[PresenceAppFieldSize];
    char streamBuffer[32];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(json, length);

    return detail::service::json::Parse(handler, stream, cancelable).IsSuccess();
}

}}}}}
