﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerFriendRequestResourceList.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

FriendRequestResourceListHandler::FriendRequestResourceListHandler() NN_NOEXCEPT :
    m_Uid(nn::account::InvalidUid),
    m_ListType(RequestListType_Faced),
    m_OutCount(0),
    m_OutRequests(nullptr),
    m_Count(0),
    m_Index(0),
    m_ReadCount(0)
{
}

void FriendRequestResourceListHandler::Initialize(int* outCount, FriendRequestImpl* outRequests,
    const nn::account::Uid& uid, RequestListType listType, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outRequests);
    NN_SDK_REQUIRES(count > 0);

    m_OutCount = outCount;
    m_OutRequests = outRequests;

    m_Uid = uid;
    m_ListType = listType;
    m_Count = count;

    std::memset(m_OutRequests, 0, sizeof (FriendRequestImpl) * m_Count);
}

int FriendRequestResourceListHandler::GetUnreadCount() const NN_NOEXCEPT
{
    return m_Index - m_ReadCount;
}

int FriendRequestResourceListHandler::GetReadCount() const NN_NOEXCEPT
{
    return m_ReadCount;
}

bool FriendRequestResourceListHandler::OnStartObject(const detail::service::json::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.items[*]"))
    {
        std::memset(&m_Request, 0, sizeof (m_Request));
        std::memset(&m_Bits, 0, sizeof (m_Bits));

        m_Request.data.uid = m_Uid;
        m_Request.data.requestListType = m_ListType;
    }

    return true;
}

bool FriendRequestResourceListHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    if (jsonPath.Match("$.items[*]"))
    {
        if (m_Bits.requestId && m_Bits.accountId && m_Bits.nickname && m_Bits.profileImageUrl &&
            m_Bits.requestType && m_Bits.requestStatus)
        {
            m_Request.data.isValid = true;

            if (m_Index < m_Count)
            {
                std::memcpy(&m_OutRequests[m_Index], &m_Request, sizeof (FriendRequestImpl));
                m_Index++;
            }
        }
    }

    return true;
}

bool FriendRequestResourceListHandler::OnEndArray(const detail::service::json::JsonPath& jsonPath, size_t numElements) NN_NOEXCEPT
{
    NN_UNUSED(numElements);

    if (jsonPath.Compare("$.items[]"))
    {
        *m_OutCount = m_Index;
    }

    return true;
}

bool FriendRequestResourceListHandler::OnBoolean(const detail::service::json::JsonPath& jsonPath, bool value) NN_NOEXCEPT
{
    if (m_ListType == RequestListType_Received)
    {
        if (!m_Bits.read &&
            jsonPath.Match("$.items[*].extras.receiver.read"))
        {
            m_Request.data.read = value;
            m_Bits.read = true;

            if (value)
            {
                m_ReadCount++;
            }
        }
    }

    return true;
}

bool FriendRequestResourceListHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.registeredTime &&
        jsonPath.Match("$.items[*].createdAt"))
    {
        m_Request.data.routeInfo.time.value = value.s;
        m_Bits.registeredTime = true;
    }

    return true;
}

bool FriendRequestResourceListHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(isOverflowed);

    if (!m_Bits.requestId &&
        jsonPath.Match("$.items[*].id"))
    {
        RequestId requestId;

        if (detail::util::HexToNumber(&requestId.value, value))
        {
            m_Request.data.requestId = requestId;
            m_Bits.requestId = true;
        }
    }
    else if (!m_Bits.requestType &&
        jsonPath.Match("$.items[*].channels[0]")) // 最初に出現したチャンネルだけで十分。
    {
        m_Request.data.requestType = ParameterConverter::ConvertRequestType(value);
        m_Bits.requestType = true;
    }
    else if (!m_Bits.requestStatus &&
        jsonPath.Match("$.items[*].state"))
    {
        m_Request.data.requestStatus = ParameterConverter::ConvertRequestStatus(value);
        m_Bits.requestStatus = true;
    }
    else if (!m_Bits.appId &&
        jsonPath.Match("$.items[*].extras.senderAndReceiver.route:appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Request.data.routeInfo.appInfo.appId = appId;
            m_Bits.appId = true;
        }
    }
    else if (!m_Bits.presenceGroupId &&
        jsonPath.Match("$.items[*].extras.senderAndReceiver.route:appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Request.data.routeInfo.appInfo.presenceGroupId = presenceGroupId;
            m_Bits.presenceGroupId = true;
        }
    }
    else if (!m_Bits.catalogId &&
        jsonPath.Match("$.items[*].extras.senderAndReceiver.route:candidate:catalogId"))
    {
        if (ParameterConverter::ConvertExternalApplicationCatalogId(&m_Request.data.routeInfoExtra.candidate.catalogId, value))
        {
            m_Bits.catalogId = true;
        }
    }
    // receiver の情報を集める。
    else if (m_ListType == RequestListType_Sent)
    {
        if (!m_Bits.accountId &&
            jsonPath.Match("$.items[*].receiverId"))
        {
            nn::account::NetworkServiceAccountId accountId;

            if (detail::util::HexToNumber(&accountId.id, value))
            {
                m_Request.data.accountId = accountId;
                m_Bits.accountId = true;
            }
        }
        else if (!m_Bits.nickname &&
            jsonPath.Match("$.items[*].receiver.nickname"))
        {
            nn::util::Strlcpy(m_Request.data.nickname.name, value, sizeof (m_Request.data.nickname.name));
            m_Bits.nickname = true;
        }
        else if (!m_Bits.profileImageUrl &&
            jsonPath.Match("$.items[*].receiver.thumbnailUrl"))
        {
            nn::util::Strlcpy(m_Request.data.profileImageUrl.value, value,
                sizeof (m_Request.data.profileImageUrl.value));
            m_Bits.profileImageUrl = true;
        }
        else if (!m_Bits.miiName &&
            jsonPath.Match("$.items[*].extras.sender.route:nnid:miiName"))
        {
            nn::util::CopyUtf8String(m_Request.data.routeInfoExtra.nnid.miiName.value,
                sizeof (m_Request.data.routeInfoExtra.nnid.miiName.value), value, length, MiiNameLengthMax);
            m_Bits.miiName = true;
        }
        else if (!m_Bits.miiImageUrlParam &&
            jsonPath.Match("$.items[*].extras.sender.route:nnid:miiImageUrlParam"))
        {
            nn::util::Strlcpy(m_Request.data.routeInfoExtra.nnid.miiImageUrlParam.value, value,
                sizeof (m_Request.data.routeInfoExtra.nnid.miiImageUrlParam.value));
            m_Bits.miiImageUrlParam = true;
        }
        else if (!m_Bits.inAppScreenName &&
            jsonPath.Match("$.items[*].extras.sender.route:name"))
        {
            nn::util::CopyUtf8String(m_Request.data.routeInfo.name.name,
                sizeof (m_Request.data.routeInfo.name.name), value, length, InAppScreenNameLengthMax);
            m_Bits.inAppScreenName = true;
        }
        else if (!m_Bits.language &&
            jsonPath.Match("$.items[*].extras.sender.route:name:language"))
        {
            nn::util::Strlcpy(m_Request.data.routeInfo.name.language.string, value,
                sizeof (m_Request.data.routeInfo.name.language.string));
            m_Bits.language = true;
        }
    }
    // sender の情報を集める。
    else if (m_ListType == RequestListType_Received)
    {
        if (!m_Bits.accountId &&
            jsonPath.Match("$.items[*].senderId"))
        {
            nn::account::NetworkServiceAccountId accountId;

            if (detail::util::HexToNumber(&accountId.id, value))
            {
                m_Request.data.accountId = accountId;
                m_Bits.accountId = true;
            }
        }
        else if (!m_Bits.nickname &&
            jsonPath.Match("$.items[*].sender.nickname"))
        {
            nn::util::Strlcpy(m_Request.data.nickname.name, value, sizeof (m_Request.data.nickname.name));
            m_Bits.nickname = true;
        }
        else if (!m_Bits.profileImageUrl &&
            jsonPath.Match("$.items[*].sender.thumbnailUrl"))
        {
            nn::util::Strlcpy(m_Request.data.profileImageUrl.value, value,
                sizeof (m_Request.data.profileImageUrl.value));
            m_Bits.profileImageUrl = true;
        }
        else if (!m_Bits.miiName &&
            jsonPath.Match("$.items[*].extras.senderAndReceiver.route:nnid:miiName"))
        {
            nn::util::CopyUtf8String(m_Request.data.routeInfoExtra.nnid.miiName.value,
                sizeof (m_Request.data.routeInfoExtra.nnid.miiName.value), value, length, MiiNameLengthMax);
            m_Bits.miiName = true;
        }
        else if (!m_Bits.miiImageUrlParam &&
            jsonPath.Match("$.items[*].extras.senderAndReceiver.route:nnid:miiImageUrlParam"))
        {
            nn::util::Strlcpy(m_Request.data.routeInfoExtra.nnid.miiImageUrlParam.value, value,
                sizeof (m_Request.data.routeInfoExtra.nnid.miiImageUrlParam.value));
            m_Bits.miiImageUrlParam = true;
        }
        else if (!m_Bits.inAppScreenName &&
            jsonPath.Match("$.items[*].extras.senderAndReceiver.route:name"))
        {
            nn::util::CopyUtf8String(m_Request.data.routeInfo.name.name,
                sizeof (m_Request.data.routeInfo.name.name), value, length, InAppScreenNameLengthMax);
            m_Bits.inAppScreenName = true;
        }
        else if (!m_Bits.language &&
            jsonPath.Match("$.items[*].extras.senderAndReceiver.route:name:language"))
        {
            nn::util::Strlcpy(m_Request.data.routeInfo.name.language.string, value,
                sizeof (m_Request.data.routeInfo.name.language.string));
            m_Bits.language = true;
        }
    }

    return true;
} // NOLINT(impl/function_size)

}}}}}
