﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerError.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    struct ErrorTable
    {
        ErrorHandler::ErrorInfo info;
        nn::Result result;
    };

    const ErrorTable ErrorTable3xx[] =
    {
        {{}, ResultUnexpectedServerError3xx()}
    };

    const ErrorTable ErrorTable4xx[] =
    {
        {{400, "invalid_params"},                           ResultServerError400InvalidParams()},
        {{400, "invalid_request"},                          ResultServerError400InvalidRequest()},
        {{400, "invalid_operation"},                        ResultServerError400InvalidOperation()},
        {{400, "sender_friend_capacity_is_full"},           ResultServerError400SenderFriendCapacityIsFull()},
        {{400, "receiver_friend_capacity_is_full"},         ResultServerError400ReceiverFriendCapacityIsFull()},
        {{401, "invalid_token"},                            ResultServerError401InvalidToken()},
        {{403, "operation_is_not_permitted"},               ResultServerError403OperationIsNotPermitted()},
        {{403, "insufficient_scope"},                       ResultServerError403InsufficientScope()},
        {{403, "forbidden"},                                ResultServerError403Forbidden()},
        {{404, "resource_is_not_found"},                    ResultServerError404ResourceIsNotFound()},
        {{405, "method_not_allowed"},                       ResultServerError405MethodNotAllowed()},
        {{406, "not_acceptable_language"},                  ResultServerError406NotAcceptableLanguage()},
        {{409, "resource_already_exists"},                  ResultServerError409ResourceAlreadyExists()},
        {{409, "resource_duplicated"},                      ResultServerError409ResourceDuplicated()},

        {{400, "receiver_must_be_different_from_sender"},   ResultOwnNetworkServiceAccountSpecified()},
        {{400, "could_not_retrieve_sender"},                ResultOwnNetworkServiceAccountNotExistsInFriendRequest()},
        {{400, "could_not_retrieve_receiver"},              ResultNetworkServiceAccountNotExistsInFriendRequest()},
        {{400, "sender_and_receiver_are_already_friends"},  ResultAlreadyFriend()},

        {{400, "invalid_friend_code_format"},               ResultInvalidFriendCodeFormat()},

        {{400, "target_user_must_be_different_from_user"},  ResultOwnNetworkServiceAccountSpecified()},
        {{404, "deleted_user"},                             ResultNetworkServiceAccountIsDeleted()},

        {{400, "could_not_retrieve_target_user"},           ResultNetworkServiceAccountNotExistsInBlock()},
        {{409, "sender_blocks_receiver"},                   ResultBlockedByMe()},
        {{409, "user_block_capacity_is_full"},              ResultBlockListLimitReached()},

        {{412, "precondition_failed"},                      ResultPreconditionFailed()},
        {{422, "friend_code_unregenerable_state"},          ResultDisallowFriendCodeIssue()},

        {{}, ResultUnexpectedServerError4xx()}
    };

    const ErrorTable ErrorTable5xx[] =
    {
        {{500, "internal_server_error"},            ResultServerError500InternalServerError()},

        {{}, ResultUnexpectedServerError5xx()}
    };
}

namespace
{
    nn::Result ConvertServerError(const ErrorHandler::ErrorInfo& info) NN_NOEXCEPT
    {
        const ErrorTable* errorTable = nullptr;

        switch (info.status / 100)
        {
        case 2:
            NN_RESULT_SUCCESS;
        case 3:
            errorTable = ErrorTable3xx;
            break;
        case 4:
            errorTable = ErrorTable4xx;
            break;
        case 5:
            errorTable = ErrorTable5xx;
            break;
        default:
            NN_RESULT_THROW(ResultUnexpectedServerErrorXxx());
        }

        while (errorTable->info.status > 0)
        {
            if (errorTable->info.status == info.status &&
                nn::util::Strncmp(errorTable->info.errorCode, info.errorCode, sizeof (info.errorCode)) == 0)
            {
                break;
            }

            errorTable++;
        }

        return errorTable->result;
    }
}

ErrorHandler::ErrorHandler() NN_NOEXCEPT
{
    m_Info.status = 200;
    m_Info.errorCode[0] = '\0';

    std::memset(&m_Bits, 0, sizeof (m_Bits));
}

nn::Result ErrorHandler::GetResult() const NN_NOEXCEPT
{
    if (!m_Bits.status && !m_Bits.errorCode)
    {
        NN_RESULT_THROW(ResultIvalidResponse());
    }

    if (m_Bits.status && m_Bits.errorCode)
    {
        return ConvertServerError(m_Info);
    }

    NN_RESULT_THROW(ResultUnexpectedServerErrorXxx());
}

bool ErrorHandler::OnInteger(const detail::service::json::JsonPath& jsonPath,
    const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.status &&
        jsonPath.Compare("$.status"))
    {
        m_Info.status = static_cast<int32_t>(value.s);
        m_Bits.status = true;
    }

    return true;
}

bool ErrorHandler::OnString(const detail::service::json::JsonPath& jsonPath,
    const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(length);
    NN_UNUSED(isOverflowed);

    if (!m_Bits.errorCode &&
        jsonPath.Compare("$.errorCode"))
    {
        nn::util::Strlcpy(m_Info.errorCode, value, sizeof (m_Info.errorCode));
        m_Bits.errorCode = true;
    }
    else if (!m_Bits.title &&
        jsonPath.Compare("$.title"))
    {
        NN_DETAIL_FRIENDS_WARN("[friends] Error: message=%s%s\n", value, isOverflowed ? " ..." : "");
        m_Bits.title = true;
    }

    return true;
}

}}}}}
