﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/handler/friends_HandlerBlockedUserResourceList.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

BlockedUserResourceListHandler::BlockedUserResourceListHandler() NN_NOEXCEPT :
    m_Resources(nullptr),
    m_Count(0)
{
}

nn::Result BlockedUserResourceListHandler::Initialize(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size >= sizeof (BlockedUserResource) * BlockedUserCountMax);

    NN_RESULT_THROW_UNLESS(size >= sizeof (BlockedUserResource) * BlockedUserCountMax, ResultInvalidArgument());

    m_Resources = reinterpret_cast<BlockedUserResource*>(buffer);
    m_Count = 0;

    NN_RESULT_SUCCESS;
}

const BlockedUserResource* BlockedUserResourceListHandler::GetResourceList() const NN_NOEXCEPT
{
    return m_Resources;
}

int BlockedUserResourceListHandler::GetResourceCount() const NN_NOEXCEPT
{
    return m_Count;
}

bool BlockedUserResourceListHandler::OnStartObject(const detail::service::json::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.items[*]"))
    {
        if (m_Count < BlockedUserCountMax)
        {
            std::memset(&m_Resources[m_Count], 0, sizeof (BlockedUserResource));
            std::memset(&m_Bits, 0, sizeof (m_Bits));
        }
    }

    return true;
}

bool BlockedUserResourceListHandler::OnEndObject(const detail::service::json::JsonPath& jsonPath, size_t numObjects) NN_NOEXCEPT
{
    if (jsonPath.Match("$.items[*]"))
    {
        if (m_Bits.accountId && m_Bits.nickname && m_Bits.profileImageUrl)
        {
            if (m_Count < BlockedUserCountMax)
            {
                m_Count++;
            }
        }
    }

    return true;
}

bool BlockedUserResourceListHandler::OnInteger(const detail::service::json::JsonPath& jsonPath, const detail::service::json::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_Bits.routeTime &&
        jsonPath.Match("$.items[*].createdAt"))
    {
        m_Resources[m_Count].routeInfo.time.value = value.s;
        m_Bits.routeTime = true;
    }

    return true;
}

bool BlockedUserResourceListHandler::OnString(const detail::service::json::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    if (!m_Bits.accountId &&
        jsonPath.Match("$.items[*].targetUserId"))
    {
        nn::account::NetworkServiceAccountId accountId;

        if (detail::util::HexToNumber(&accountId.id, value))
        {
            m_Resources[m_Count].accountId = accountId;
            m_Bits.accountId = true;
        }
    }
    else if (!m_Bits.nickname &&
        jsonPath.Match("$.items[*].targetUser.nickname"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].nickname.name, value, sizeof (m_Resources[m_Count].nickname.name));
        m_Bits.nickname = true;
    }
    else if (!m_Bits.profileImageUrl &&
        jsonPath.Match("$.items[*].targetUser.thumbnailUrl"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].profileImageUrl.value, value,
            sizeof (m_Resources[m_Count].profileImageUrl.value));
        m_Bits.profileImageUrl = true;
    }
    else if (!m_Bits.blockReason &&
        jsonPath.Match("$.items[*].extras.self.reason"))
    {
        m_Resources[m_Count].blockReason = ParameterConverter::ConvertBlockReason(value);
        m_Bits.blockReason = true;
    }
    else if (!m_Bits.routeAppId &&
        jsonPath.Match("$.items[*].extras.self.route:appInfo:appId"))
    {
        nn::ApplicationId appId;

        if (detail::util::HexToNumber(&appId.value, value))
        {
            m_Resources[m_Count].routeInfo.appInfo.appId = appId;
            m_Bits.routeAppId = true;
        }
    }
    else if (!m_Bits.routePresenceGroupId &&
        jsonPath.Match("$.items[*].extras.self.route:appInfo:presenceGroupId"))
    {
        Bit64 presenceGroupId;

        if (detail::util::HexToNumber(&presenceGroupId, value))
        {
            m_Resources[m_Count].routeInfo.appInfo.presenceGroupId = presenceGroupId;
            m_Bits.routePresenceGroupId = true;
        }
    }
    else if (!m_Bits.routeName &&
        jsonPath.Match("$.items[*].extras.self.route:name"))
    {
        nn::util::CopyUtf8String(m_Resources[m_Count].routeInfo.name.name,
            sizeof (m_Resources[m_Count].routeInfo.name.name), value, length, InAppScreenNameLengthMax);
        m_Bits.routeName = true;
    }
    else if (!m_Bits.routeNameLanguage &&
        jsonPath.Match("$.items[*].extras.self.route:name:language"))
    {
        nn::util::Strlcpy(m_Resources[m_Count].routeInfo.name.language.string, value,
            sizeof (m_Resources[m_Count].routeInfo.name.language.string));
        m_Bits.routeNameLanguage = true;
    }

    return true;
}

}}}}}
