﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_WebApi.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerError.h>
#include <nn/friends/detail/service/json/friends_JsonErrorHandler.h>
#include <nn/friends/detail/service/json/friends_JsonEventHandlerForReader.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    struct WebApiParam
    {
        detail::service::json::JsonEventHandlerForReader* reader;
        detail::service::json::JsonEventHandler* handler;
        detail::service::json::JsonEventHandler* errorHandler;
        int statusCode;
    };
}

namespace
{
    bool WebApiResponseHeaderEndCallback(int statusCode, void* param) NN_NOEXCEPT
    {
        WebApiParam* p = static_cast<WebApiParam*>(param);

        if (statusCode / 100 == 2)
        {
            p->reader->SetHandler(p->handler);
        }
        else
        {
            p->reader->SetHandler(p->errorHandler);
        }
        p->statusCode = statusCode;

        return true;
    }

    nn::Result HandleStatusCode(int statusCode) NN_NOEXCEPT
    {
        int category = statusCode / 100;

        if (category == 3)
        {
            switch (statusCode)
            {
            case 302:
                NN_RESULT_THROW(ResultUnexpectedServerError302());
            default:
                NN_RESULT_THROW(ResultUnexpectedServerError3xx());
            }
        }
        if (category == 4)
        {
            switch (statusCode)
            {
            case 400:
                NN_RESULT_THROW(ResultUnexpectedServerError400());
            case 403:
                NN_RESULT_THROW(ResultUnexpectedServerError403());
            case 404:
                NN_RESULT_THROW(ResultUnexpectedServerError404());
            default:
                NN_RESULT_THROW(ResultUnexpectedServerError4xx());
            }
        }
        if (category == 5)
        {
            switch (statusCode)
            {
            case 500:
                NN_RESULT_THROW(ResultUnexpectedServerError500());
            case 503:
                NN_RESULT_THROW(ResultUnexpectedServerError503());
            case 504:
                NN_RESULT_THROW(ResultUnexpectedServerError504());
            default:
                NN_RESULT_THROW(ResultUnexpectedServerError5xx());
            }
        }

        NN_RESULT_THROW(ResultUnexpectedServerErrorXxx());
    }
}

const char* WebApi::GetFqdn() NN_NOEXCEPT
{
    return "e0d67c509fb203858ebcb2fe3f88c2aa.baas.nintendo.com";
}

nn::Result WebApi::Call(detail::service::json::JsonEventHandler& okHandler, detail::service::json::JsonEventHandler& errorHandler,
    detail::service::json::JsonHttpInputStream& stream, const nn::account::Uid& uid, const detail::service::util::Cancelable& cancelable,
    bool isAuthenticationRequired) NN_NOEXCEPT
{
    // スタックの消費が大きいため、リクエストヘッダへの追加後はすぐにスタックを開放させる。
    if (isAuthenticationRequired)
    {
        Account::NetworkServiceAccessToken accessToken;
        NN_RESULT_DO(Account::GetNetworkServiceAccessToken(&accessToken, uid));

        NN_RESULT_DO(stream.AddRequestHeader("Authorization: Bearer %s", accessToken.value));
    }

    detail::service::json::JsonEventHandlerForReader handlerForReader;

    handlerForReader.SetCancelable(&cancelable);
    handlerForReader.SetOverflowDetector(stream.GetOverflowDetector());

    // 文字列データを格納するバッファ。取得したい文字列データの最大サイズ＋αがちょうどよいサイズ。
    char stringBuffer[6144];
    // 入力ストリームのワークバッファ。サイズが大きいほど、読み込み性能は高い。
    char streamBuffer[1024];

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.SetCancelable(&cancelable);

    WebApiParam param =
    {
        &handlerForReader,
        &okHandler,
        &errorHandler
    };

    stream.RegisterResponseHeaderEndCallback(WebApiResponseHeaderEndCallback, &param);

    detail::service::json::NullAllocator allocator;

    const unsigned int ParseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;

    RAPIDJSON_NAMESPACE::ParseResult result =
        detail::service::json::JsonReader(&allocator, 1).Parse<ParseFlags>(stream, handlerForReader);

    // エラーの優先度は以下の通り。
    // キャンセル > 通信 > パーサー > エラーハンドラ

    if (cancelable.IsCanceled())
    {
        NN_RESULT_THROW(ResultCanceled());
    }

    NN_RESULT_DO(stream.GetLastError());

    int category = param.statusCode / 100;

    if (category != 2 && ResultIvalidResponse::Includes(errorHandler.GetResult()))
    {
        NN_RESULT_DO(HandleStatusCode(param.statusCode));
    }

    NN_RESULT_DO(detail::service::json::HandleJsonError(result));

    if (category != 2)
    {
        NN_RESULT_DO(errorHandler.GetResult());
    }

    NN_RESULT_SUCCESS;
}

nn::Result WebApi::Call(detail::service::json::JsonEventHandler& okHandler,
    detail::service::json::JsonHttpInputStream& stream, const nn::account::Uid& uid, const detail::service::util::Cancelable& cancelable,
    bool isAuthenticationRequired) NN_NOEXCEPT
{
    ErrorHandler errorHandler;
    return Call(okHandler, errorHandler, stream, uid, cancelable, isAuthenticationRequired);
}

}}}}}
