﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_UuidManager.h>
#include <nn/util/util_UuidApi.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* FileName = "uuid.bin";
}

UuidManager::UuidManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_CurrentUid(nn::account::InvalidUid)
{
}

nn::Result UuidManager::Get(nn::util::Uuid* outUuid, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outUuid);
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    *outUuid = m_Uuid;

    NN_RESULT_SUCCESS;
}

void UuidManager::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_CurrentUid = nn::account::InvalidUid;
}

nn::Result UuidManager::Load(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadImpl(uid))
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_DO(Generate(uid));
        }
    NN_RESULT_END_TRY;

    m_CurrentUid = uid;

    NN_RESULT_SUCCESS;
}

nn::Result UuidManager::LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT
{
    if (uid == m_CurrentUid)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    char path[64];
    AccountStorageManager::GetInstance().MakePath(path, sizeof (path), FileName);

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, &m_Uuid, sizeof (m_Uuid)));

    NN_RESULT_SUCCESS;
}

nn::Result UuidManager::Generate(const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::util::Uuid uuid = nn::util::GenerateUuid();

    NN_RESULT_DO(Save(uid, uuid));

    m_Uuid = uuid;

    NN_RESULT_SUCCESS;
}

nn::Result UuidManager::Save(const nn::account::Uid& uid, const nn::util::Uuid& uuid) NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    {
        char path[64];
        AccountStorageManager::GetInstance().MakePath(path, sizeof (path), FileName);

        NN_RESULT_TRY(nn::fs::DeleteFile(path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (uuid.data)));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, uuid.data, sizeof (uuid.data),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    NN_RESULT_SUCCESS;
}

}}}}}
