﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_ImageManager.h>
#include <nn/crypto/crypto_Md5Generator.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* RecordFilePath = "friends-sys:/image_record.bin";
}

ImageManager::ImageManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_Count(0),
    m_IsLoaded(false)
{
}

nn::Result ImageManager::SaveImage(nn::account::NetworkServiceAccountId accountId, const Url& url,
        const void* image, size_t imageSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(image);
    NN_SDK_REQUIRES(imageSize > 0);

    Record record;
    CreateRecord(&record, accountId, url);

    char path[64];
    MakeImagePath(path, sizeof (path), record);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_FRIENDS_SYSTEM_STORAGE_SCOPED_LOCK();
    NN_DETAIL_FRIENDS_IMAGE_STORAGE_SCOPED_LOCK();

    NN_RESULT_DO(LoadRecord());

    bool isSuccess = false;

    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            m_IsLoaded = false;
        }
    };

    NN_RESULT_DO(DeleteImage(accountId));
    RemoveRecord(accountId);

    NN_RESULT_DO(EnsureCapacity(static_cast<int64_t>(imageSize)));

    int index = SearchRecord(accountId);

    if (index == -1)
    {
        if (m_Count == RecordCountMax)
        {
            RemoveOldRecord();
        }
        InsertRecord(record);
    }
    else
    {
        m_Records[index] = record;
    }

    NN_RESULT_DO(FileSystem::CreateFile(path, imageSize));

    {
        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, image, imageSize,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(SaveRecord());

    NN_RESULT_DO(FileSystem::Commit(NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME));
    NN_RESULT_DO(FileSystem::Commit(NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME));

    isSuccess = true;

    NN_RESULT_SUCCESS;
}

nn::Result ImageManager::LoadImage(size_t* outSize, nn::account::NetworkServiceAccountId accountId,
    void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_FRIENDS_SYSTEM_STORAGE_SCOPED_LOCK();
    NN_DETAIL_FRIENDS_IMAGE_STORAGE_SCOPED_LOCK();

    NN_RESULT_DO(LoadRecord());

    int index = SearchRecord(accountId);

    NN_RESULT_THROW_UNLESS(index != -1, ResultProfileImageCacheNotFound());
    NN_RESULT_THROW_UNLESS(!m_Records[index].isDeleted, ResultProfileImageCacheNotFound());

    char path[64];
    MakeImagePath(path, sizeof (path), m_Records[index]);

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            m_Records[index].isDeleted = true;
            NN_RESULT_THROW(ResultProfileImageCacheNotFound());
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t fileSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    NN_RESULT_THROW_UNLESS(fileSize <= static_cast<int64_t>(size), ResultOutOfResource());

    size_t read;
    NN_RESULT_DO(nn::fs::ReadFile(&read, handle, 0, buffer, size));

    *outSize = read;

    NN_RESULT_SUCCESS;
}

bool ImageManager::CanAddOrUpdate(nn::account::NetworkServiceAccountId accountId, const Url& url) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    Record record;
    CreateRecord(&record, accountId, url);

    LoadRecordWithLockStorage();

    int index = SearchRecord(accountId);

    if (index == -1)
    {
        // レコードに存在しない場合、新規追加可能。
        return true;
    }
    else
    {
        // データが存在し、URL ハッシュが異なる場合、上書き更新可能。
        if (!m_Records[index].isDeleted && std::memcmp(&record.hash, &m_Records[index].hash, sizeof (record.hash)) != 0)
        {
            return true;
        }
    }

    return false;
}

void ImageManager::MoveToTopRecords(const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(accountIds);
    NN_SDK_REQUIRES(count > 0);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    LoadRecordWithLockStorage();

    for (int i = count - 1; i >= 0; i--)
    {
        MoveToTopRecord(accountIds[i], false);
    }
}

void ImageManager::MoveToTopRecords(const FriendImpl* friends, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(friends);
    NN_SDK_REQUIRES(count > 0);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    LoadRecordWithLockStorage();

    for (int i = count - 1; i >= 0; i--)
    {
        MoveToTopRecord(friends[i].data.accountId, false);
    }
}

void ImageManager::CreateRecord(Record* outRecord, nn::account::NetworkServiceAccountId accountId, const Url& url) NN_NOEXCEPT
{
    Bit8 hash[16];

    nn::crypto::GenerateMd5Hash(hash, 16,
        url.value, static_cast<size_t>(nn::util::Strnlen(url.value, sizeof (url.value))));

    outRecord->accountId = accountId;
    outRecord->isDeleted = false;

    std::memcpy(&outRecord->hash.value, hash, sizeof (outRecord->hash.value));
}

void ImageManager::InsertRecord(const Record& record) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Count < RecordCountMax);

    m_Count = detail::service::util::ArrayAccessor::InsertToTopEntry(m_Records, m_Count, record);
}

void ImageManager::MoveToTopRecord(nn::account::NetworkServiceAccountId accountId, bool allowDeleted) NN_NOEXCEPT
{
    int index = SearchRecord(accountId);

    if (index <= 0)
    {
        return;
    }
    if (!allowDeleted && m_Records[index].isDeleted)
    {
        return;
    }

    detail::service::util::ArrayAccessor::MoveToTopEntry(m_Records, m_Count, index);
}

void ImageManager::RemoveRecord(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    m_Count = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, accountId,
        [](const Record& lhs, const nn::account::NetworkServiceAccountId& rhs)
        {
            return lhs.accountId == rhs;
        });
}

void ImageManager::RemoveOldRecord() NN_NOEXCEPT
{
    if (m_Count == 0)
    {
        return;
    }

    m_Count--;
}

nn::Result ImageManager::LoadRecordWithLockStorage() NN_NOEXCEPT
{
    if (!m_IsLoaded)
    {
        NN_DETAIL_FRIENDS_SYSTEM_STORAGE_SCOPED_LOCK();
        NN_RESULT_DO(LoadRecord());
    }

    NN_RESULT_SUCCESS;
}

nn::Result ImageManager::LoadRecord() NN_NOEXCEPT
{
    if (m_IsLoaded)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_TRY(LoadRecordImpl())
        NN_RESULT_CATCH_ALL
        {
            m_Count = 0;
        }
    NN_RESULT_END_TRY;

    m_IsLoaded = true;

    NN_RESULT_SUCCESS;
}

nn::Result ImageManager::LoadRecordImpl() NN_NOEXCEPT
{
    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, RecordFilePath, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    if ((size % sizeof (Record)) != 0)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. ((size % sizeof (Record)) != 0)\n", RecordFilePath);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    int count = static_cast<int>(size / sizeof (Record));

    if (count > RecordCountMax)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (count > RecordCountMax)\n", RecordFilePath);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (Record) * count));
    m_Count = count;

    NN_RESULT_SUCCESS;
}

nn::Result ImageManager::SaveRecord() NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::DeleteFile(RecordFilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    if (m_Count == 0)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(FileSystem::CreateFile(RecordFilePath, sizeof (Record) * m_Count));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, RecordFilePath, nn::fs::OpenMode_Write));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (Record) * m_Count,
        nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

int ImageManager::SearchRecord(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (accountId == m_Records[i].accountId)
        {
            return i;
        }
    }

    return -1;
}

void ImageManager::MakeImagePath(char* buffer, size_t size, const Record& record) NN_NOEXCEPT
{
    nn::util::SNPrintf(buffer, size,
        "friends-image:/image/%016llx_%02x%02x%02x%02x%02x%02x%02x.jpg",
        record.accountId.id,
        record.hash.value[0], record.hash.value[1], record.hash.value[2], record.hash.value[3],
        record.hash.value[4], record.hash.value[5], record.hash.value[6]);
}

nn::Result ImageManager::EnsureCapacity(int64_t requiredSize) NN_NOEXCEPT
{
    bool isDeleted = false;

    while (NN_STATIC_CONDITION(true))
    {
        int64_t freeSpaceSize;
        NN_RESULT_DO(FileSystem::GetFreeSpaceSize(&freeSpaceSize, NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME));

        NN_DETAIL_FRIENDS_INFO("[friends] Image storage: free space size = %lld, required size = %zu.\n",
            freeSpaceSize, requiredSize);

        if (requiredSize < freeSpaceSize)
        {
            break;
        }

        NN_RESULT_TRY(DeleteOldImage())
            NN_RESULT_CATCH(ResultProfileImageCacheNotFound)
            {
                NN_RESULT_THROW(ResultOutOfResource());
            }
        NN_RESULT_END_TRY;

        isDeleted = true;
    }

    if (isDeleted)
    {
        NN_RESULT_DO(SaveRecord());

        NN_RESULT_DO(FileSystem::Commit(NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME));
        NN_RESULT_DO(FileSystem::Commit(NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME));
    }

    NN_RESULT_SUCCESS;
}

nn::Result ImageManager::DeleteImage(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    int index = SearchRecord(accountId);

    if (index == -1)
    {
        NN_RESULT_SUCCESS;
    }

    char path[64];
    MakeImagePath(path, sizeof (path), m_Records[index]);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    m_Records[index].isDeleted = true;

    NN_RESULT_SUCCESS;
}

nn::Result ImageManager::DeleteOldImage() NN_NOEXCEPT
{
    if (m_Count == 0)
    {
        NN_RESULT_SUCCESS;
    }

    for (int i = m_Count - 1; i >= 0; i--)
    {
        if (!m_Records[i].isDeleted)
        {
            char path[64];
            MakeImagePath(path, sizeof (path), m_Records[i]);

            NN_RESULT_TRY(nn::fs::DeleteFile(path))
                NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
                {
                }
            NN_RESULT_END_TRY;

            m_Records[i].isDeleted = true;
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultProfileImageCacheNotFound());
}

}}}}}
