﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_FriendRequestCountManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

FriendRequestCountManager::FriendRequestCountManager() NN_NOEXCEPT :
    m_Mutex(true)
{
    std::memset(m_Records, 0, sizeof (m_Records));

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        m_Records[i].uid = nn::account::InvalidUid;
    }
}

nn::Result FriendRequestCountManager::GetCount(int* outUnreadCount, int* outReadCount,
    const nn::account::Uid& uid) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outUnreadCount);
    NN_SDK_REQUIRES_NOT_NULL(outReadCount);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNetworkServiceAccountNotExistsGeneral());

    *outUnreadCount = m_Records[index].unreadCount;
    *outReadCount = m_Records[index].readCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendRequestCountManager::SetCount(const nn::account::Uid& uid, int unreadCount, int readCount) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNetworkServiceAccountNotExistsGeneral());

    bool isReceived = false;

    // 未読のフレンド申請が増えた場合、通知する。
    if (m_Records[index].unreadCount < unreadCount)
    {
        isReceived = true;
    }

    m_Records[index].unreadCount = unreadCount;
    m_Records[index].readCount = readCount;

    if (isReceived)
    {
        NotificationEventHandler::GetInstance().NotifyFriendRequestReceived(uid);
    }

    NN_RESULT_SUCCESS;
}

void FriendRequestCountManager::NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = -1;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == nn::account::InvalidUid)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        return;
    }

    std::memset(&m_Records[index], 0, sizeof (Record));
    m_Records[index].uid = uid;
}

void FriendRequestCountManager::NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    m_Records[index].uid = nn::account::InvalidUid;
}

void FriendRequestCountManager::NotifyNetworkServiceAccountAvailabilityChanged(const nn::account::Uid& uid, bool isAvailable) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    if (!isAvailable)
    {
        m_Records[index].unreadCount = 0;
        m_Records[index].readCount = 0;
    }
}

int FriendRequestCountManager::FindUser(const nn::account::Uid& uid) const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == uid)
        {
            return i;
        }
    }

    return -1;
}

}}}}}
