﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_ForegroundTaskManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

ForegroundTaskManager::ForegroundTaskManager() NN_NOEXCEPT :
    m_Mutex(true)
{
    for (auto& event : m_TaskEvents)
    {
        event.allocated = false;
    }
}

ForegroundTaskManager::~ForegroundTaskManager() NN_NOEXCEPT
{
    RemoveAll();

    for (auto& event : m_TaskEvents)
    {
        if (event.allocated)
        {
            nn::os::FinalizeEvent(&event.runnableEvent);
        }
    }
}

nn::os::EventType* ForegroundTaskManager::AcquireRunnableTaskEvent() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (auto& event : m_TaskEvents)
    {
        if (!event.allocated)
        {
            nn::os::InitializeEvent(&event.runnableEvent, true, nn::os::EventClearMode_ManualClear);
            event.allocated = true;

            return &event.runnableEvent;
        }
    }

    NN_SDK_ASSERT(0, "ForegroundTaskManager::AcquireRunnableTaskEvent allocation failed.");

    return nullptr;
}

void ForegroundTaskManager::ReleaseRunnableTaskEvent(nn::os::EventType* runnableEvent) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (auto& event : m_TaskEvents)
    {
        if (runnableEvent == &event.runnableEvent)
        {
            nn::os::FinalizeEvent(&event.runnableEvent);
            event.allocated = false;
        }
    }
}

void ForegroundTaskManager::Add(Task* task) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(task);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    task->SetStatus(Task::Status_Waiting);
    m_TaskQueue.push_back(*task);

    SignalRunnableTaskEvents();
}

void ForegroundTaskManager::Remove(Task* task) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(task);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (auto it = m_TaskQueue.begin(); it != m_TaskQueue.end(); ++it)
    {
        Task* p = &(*it);

        if (p->GetContextId() == task->GetContextId())
        {
            m_TaskQueue.erase(it);
            return;
        }
    }
}

void ForegroundTaskManager::RemoveAll() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    while (m_TaskQueue.begin() != m_TaskQueue.end())
    {
        auto it = m_TaskQueue.begin();

        m_TaskQueue.erase(it);
    }
}

bool ForegroundTaskManager::RunRunnable() NN_NOEXCEPT
{
    uint64_t contextId = 0;
    Task* runnable = nullptr;

    {
        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        for (auto& task : m_TaskQueue)
        {
            if (task.GetStatus() == Task::Status_Waiting)
            {
                runnable = &task;
                runnable->SetStatus(Task::Status_Running);
                contextId = runnable->GetContextId();
                break;
            }
        }
    }

    if (!runnable)
    {
        return false;
    }

    runnable->Run();

    {
        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        for (auto it = m_TaskQueue.begin(); it != m_TaskQueue.end(); ++it)
        {
            Task* p = &(*it);

            if (p->GetContextId() == contextId)
            {
                m_TaskQueue.erase(it);
                break;
            }
        }

        if (HasRunnable())
        {
            SignalRunnableTaskEvents();
        }
    }

    return true;
}

bool ForegroundTaskManager::HasRunnable() NN_NOEXCEPT
{
    for (auto& task : m_TaskQueue)
    {
        if (task.GetStatus() == Task::Status_Waiting)
        {
            return true;
        }
    }

    return false;
}

void ForegroundTaskManager::SignalRunnableTaskEvents() NN_NOEXCEPT
{
    for (auto& event : m_TaskEvents)
    {
        if (event.allocated)
        {
            nn::os::SignalEvent(&event.runnableEvent);
        }
    }
}

}}}}}
