﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_FileSystem.h>
#include <nn/friends/detail/service/core/friends_FileSystemConfig.h>
#include <nn/friends/detail/service/friends_Common.h>
#include <nn/fs/fs_FileSystemPrivate.h>

#define ENABLE_HOSTFS 0 // NOLINT(preprocessor/const)

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    nn::os::SdkMutexType g_Mutexes[2] =
    {
        NN_OS_SDK_MUTEX_INITIALIZER(),
        NN_OS_SDK_MUTEX_INITIALIZER()
    };
}

namespace
{
    nn::Result MountSystemSaveData(const char* name, nn::fs::SystemSaveDataId id, int64_t size, int64_t journalSize) NN_NOEXCEPT
    {
#if ENABLE_HOSTFS == 0

        NN_RESULT_TRY(nn::fs::MountSystemSaveData(name, id))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_DETAIL_FRIENDS_INFO("[friends] I create new system save data. (%s)\n", name);

                NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, size, journalSize, 0));
                NN_RESULT_DO(nn::fs::MountSystemSaveData(name, id));
            }
        NN_RESULT_END_TRY;

#else

        char root[128] = {};
        nn::util::SNPrintf(root, sizeof (root), "C:\\siglo\\%s", name);

        NN_RESULT_DO(nn::fs::MountHost(name, root));

#endif

        NN_RESULT_SUCCESS;
    }

#if defined (NN_DETAIL_ENABLE_SDK_ASSERT)

    bool IsMounted(const char* mountName) NN_NOEXCEPT
    {
        if (nn::util::Strncmp(mountName, NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME, nn::fs::MountNameLengthMax + 1) == 0)
        {
            return g_Mutexes[0].IsLockedByCurrentThread();
        }
        else if (nn::util::Strncmp(mountName, NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME, nn::fs::MountNameLengthMax + 1) == 0)
        {
            return g_Mutexes[1].IsLockedByCurrentThread();
        }

        return false;
    }

#endif
}

nn::Result FileSystem::Mount(const char* mountName) NN_NOEXCEPT
{
    bool isSuccess = false;

    if (nn::util::Strncmp(mountName, NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME, nn::fs::MountNameLengthMax + 1) == 0)
    {
        g_Mutexes[0].lock();

        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                g_Mutexes[0].unlock();
            }
        };

        NN_RESULT_DO(MountSystemSaveData(NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME, 0x8000000000000081,
            SystemStorageSize, SystemStorageJournalSize));

        isSuccess = true;
    }
    else if (nn::util::Strncmp(mountName, NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME, nn::fs::MountNameLengthMax + 1) == 0)
    {
        g_Mutexes[1].lock();

        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                g_Mutexes[1].unlock();
            }
        };

        NN_RESULT_DO(MountSystemSaveData(NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME, 0x8000000000000082,
            ImageStorageSize, ImageStorageJournalSize));

        isSuccess = true;
    }
    else
    {
        NN_ABORT("");
    }

    NN_RESULT_SUCCESS;
}

void FileSystem::Unmount(const char* mountName) NN_NOEXCEPT
{
    if (nn::util::Strncmp(mountName, NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME, nn::fs::MountNameLengthMax + 1) == 0)
    {
        NN_SDK_ASSERT(g_Mutexes[0].IsLockedByCurrentThread());

        nn::fs::Unmount(mountName);

        g_Mutexes[0].unlock();
    }
    else if (nn::util::Strncmp(mountName, NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME, nn::fs::MountNameLengthMax + 1) == 0)
    {
        NN_SDK_ASSERT(g_Mutexes[1].IsLockedByCurrentThread());

        nn::fs::Unmount(mountName);

        g_Mutexes[1].unlock();
    }
    else
    {
        NN_ABORT("");
    }
}

nn::Result FileSystem::GetFreeSpaceSize(int64_t* outSize, const char* mountName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);

    NN_SDK_ASSERT(IsMounted(mountName));

#if ENABLE_HOSTFS == 0

    char root[nn::fs::MountNameLengthMax + 3] = {};
    nn::util::SNPrintf(root, sizeof (root), "%s:/", mountName);

    NN_RESULT_DO(nn::fs::GetFreeSpaceSize(outSize, root));

#else

    if (nn::util::Strncmp(mountName, NN_DETAIL_FRIENDS_SYSTEM_MOUNT_NAME, nn::fs::MountNameLengthMax + 1) == 0)
    {
        *outSize = SystemStorageSize;
    }
    else if (nn::util::Strncmp(mountName, NN_DETAIL_FRIENDS_IMAGE_MOUNT_NAME, nn::fs::MountNameLengthMax + 1) == 0)
    {
        *outSize = ImageStorageSize;
    }
    else
    {
        NN_ABORT("");
    }

#endif

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::Commit(const char* mountName) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsMounted(mountName));

#if ENABLE_HOSTFS == 0

    NN_RESULT_DO(nn::fs::CommitSaveData(mountName));

#else

    NN_UNUSED(mountName);

#endif

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::CreateDirectoryRecursively(const char* path, bool isFilePath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    char currentPath[nn::fs::EntryNameLengthMax + 1] = {};
    bool root = true;

    char c = path[0];

    for (int i = 0; path[i] != '\0'; i++)
    {
        c = path[i];

        if (c == '/')
        {
            if (!root)
            {
                NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
                    NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                    {
                    }
                NN_RESULT_END_TRY;
            }
            else
            {
                root = false;
            }
        }

        currentPath[i] = c;
    }

    if (!root && !isFilePath && c != '/')
    {
        NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    return nn::fs::DeleteDirectoryRecursively(path);
}

nn::Result FileSystem::CreateFile(const char* path, int64_t size, bool ignoreIfExists) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    NN_RESULT_TRY(nn::fs::CreateFile(path, size))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(CreateDirectoryRecursively(path, true));
            NN_RESULT_DO(nn::fs::CreateFile(path, size));
        }
        NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
        {
            if (!ignoreIfExists)
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

}}}}}
