﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_FacedFriendRequestManager.h>
#include <nn/crypto.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* FileName = "faced.bin";

    const size_t HashKeyCount = 8;
    const size_t HashKeySize = 32;

    // HMAC-SHA256 用鍵
    const Bit8 g_HashKey[HashKeyCount][HashKeySize] =
    {
        {
            0x48, 0x1E, 0x76, 0xBB, 0x9B, 0xF8, 0x45, 0xD1, 0x90, 0x61, 0x67, 0x0C, 0x60, 0x3A, 0xBF, 0x80,
            0x9F, 0x0A, 0xA1, 0x10, 0x05, 0xC3, 0x43, 0x7F, 0x95, 0x86, 0x76, 0x4D, 0x6F, 0x7E, 0x8C, 0xC7
        },
        {
            0xF6, 0x09, 0xB7, 0x34, 0x7D, 0xF3, 0x49, 0xE5, 0x80, 0xF8, 0xDA, 0xA1, 0xCB, 0x6F, 0x36, 0x8F,
            0xF7, 0x46, 0x5A, 0x4E, 0xCA, 0x36, 0x41, 0x81, 0x84, 0x82, 0xC0, 0x8A, 0x88, 0xCC, 0x4D, 0x2E
        },
        {
            0x0A, 0xBB, 0x4E, 0xC2, 0xEE, 0xB6, 0x42, 0xFC, 0xAE, 0x3D, 0x7F, 0xD2, 0xF2, 0x9A, 0xC1, 0xDD,
            0x3D, 0x35, 0xDD, 0x7D, 0x93, 0x74, 0x4B, 0xE1, 0xB4, 0x3E, 0x45, 0xCD, 0x0E, 0x96, 0x56, 0xB1
        },
        {
            0x23, 0x98, 0xCB, 0x0E, 0x0A, 0x8E, 0x4B, 0xA1, 0xB2, 0x6A, 0xAE, 0xC7, 0xD9, 0x2A, 0xFE, 0x92,
            0x56, 0x2E, 0xBB, 0x44, 0x94, 0x2C, 0x45, 0x66, 0x82, 0xC8, 0x39, 0xBD, 0xEC, 0xBB, 0x4B, 0xD0
        },
        {
            0x58, 0xE9, 0xF4, 0x1F, 0xFC, 0x07, 0x4A, 0x78, 0x82, 0x47, 0x66, 0x11, 0x6D, 0x3B, 0x22, 0x8A,
            0x33, 0x55, 0x78, 0xFE, 0xA5, 0x16, 0x4C, 0x65, 0x95, 0x48, 0x4F, 0x09, 0xD5, 0xF8, 0xAC, 0x0C
        },
        {
            0xBB, 0x6B, 0xE2, 0x4E, 0xE1, 0x6D, 0x4A, 0x9D, 0xAC, 0x0B, 0x01, 0x3C, 0x46, 0xB9, 0x5D, 0x33,
            0x70, 0xEE, 0xB6, 0x84, 0x3E, 0x73, 0x44, 0x4B, 0xA4, 0xE6, 0x59, 0xF0, 0x06, 0x21, 0x60, 0x6F
        },
        {
            0xE3, 0x57, 0xF4, 0x46, 0xE8, 0x61, 0x4B, 0xB1, 0xA7, 0xFF, 0x31, 0x61, 0x48, 0xFF, 0x84, 0x3B,
            0xE4, 0x24, 0x8C, 0x55, 0x2D, 0x5A, 0x45, 0x9C, 0xA7, 0x53, 0x4C, 0x6C, 0x58, 0x60, 0xEC, 0x17
        },
        {
            0xFA, 0xA9, 0xAD, 0x17, 0x11, 0x15, 0x4C, 0x66, 0x96, 0x73, 0x21, 0xBA, 0x46, 0xB6, 0x7C, 0x7B,
            0xCF, 0xD0, 0x56, 0xC2, 0xEB, 0x07, 0x48, 0x7B, 0x9E, 0xCB, 0x43, 0xD9, 0xE2, 0xDE, 0xE5, 0x7F
        }
    };
}

FacedFriendRequestManager::FacedFriendRequestManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_CurrentUid(nn::account::InvalidUid),
    m_Count(0)
{
}

nn::Result FacedFriendRequestManager::GetFacedFriendRequestRegistrationKey(FacedFriendRequestRegistrationKey* outKey,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outKey);

    nn::account::NetworkServiceAccountId accountId;
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, uid));

    FacedFriendRequestRegistrationKeyData data = {};

    // バージョンは 1 で固定。
    data.version = 1;
    // 1=NX とする。
    data.platform = 1;
    // 使用するキーを取得毎に変える。
    data.hashKeyIndex = static_cast<Bit8>(detail::service::util::GetRandom() % HashKeyCount);

    data.accountId = accountId;

    // 前半 32 バイトのハッシュをとる。
    nn::crypto::GenerateHmacSha256Mac(data.hash, sizeof (data.hash), &data, 32,
        &g_HashKey[data.hashKeyIndex], HashKeySize);

    std::memcpy(outKey, &data, sizeof (data));

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::AddFacedFriendRequest(const nn::account::Uid& uid,
    const FacedFriendRequestRegistrationKey& key, const nn::account::Nickname& nickname,
    const void* image, size_t imageSize) NN_NOEXCEPT
{
    ApplicationInfo appInfo = {};
    InAppScreenName inAppScreenName = {};
    InAppScreenName myInAppScreenName = {};

    return AddFacedFriendRequest(uid, key, nickname, image, imageSize, appInfo, inAppScreenName, myInAppScreenName);
}

nn::Result FacedFriendRequestManager::AddFacedFriendRequest(const nn::account::Uid& uid,
    const FacedFriendRequestRegistrationKey& key, const nn::account::Nickname& nickname,
    const void* image, size_t imageSize,
    const ApplicationInfo& appInfo, const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(image);
    NN_SDK_REQUIRES(imageSize > 0);

    NN_RESULT_THROW_UNLESS(imageSize <= ProfileImageSizeMax, ResultInvalidArgument());

    const FacedFriendRequestRegistrationKeyData* data =
        reinterpret_cast<const FacedFriendRequestRegistrationKeyData*>(&key);

    // バージョンは 1 で固定。
    NN_RESULT_THROW_UNLESS(data->version == 1,
        ResultFacedFriendRequestRegistrationKeyBroken());

    // TORIAEZU: 1=NX
    NN_RESULT_THROW_UNLESS(data->platform == 1,
        ResultFacedFriendRequestRegistrationKeyBroken());

    NN_RESULT_THROW_UNLESS(data->hashKeyIndex >= 0 && data->hashKeyIndex < HashKeyCount,
        ResultFacedFriendRequestRegistrationKeyBroken());

    Bit8 hash[32];

    // 前半 32 バイトのハッシュをとる。
    nn::crypto::GenerateHmacSha256Mac(hash, sizeof (hash), data, 32,
        &g_HashKey[data->hashKeyIndex], HashKeySize);

    NN_RESULT_THROW_UNLESS(std::memcmp(hash, data->hash, sizeof (hash)) == 0,
        ResultFacedFriendRequestRegistrationKeyBroken());

    nn::account::NetworkServiceAccountId myAccountId;
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&myAccountId, uid));

    NN_RESULT_THROW_UNLESS(data->accountId != myAccountId, ResultOwnFacedFriendRequestRegistrationKey());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    // TODO: ニックネーム検証

    NN_RESULT_DO(Load(uid));

    for (int i = 0; i < m_Count; i++)
    {
        NN_RESULT_THROW_UNLESS(data->accountId != m_Records[i].resource.accountId, ResultFacedFriendRequestAlreadyAdded());
    }

    NN_RESULT_THROW_UNLESS(m_Count < FacedFriendRequestCountMax, ResultFacedFriendRequestListLimitReached());

    m_Records[m_Count].resource.accountId = data->accountId;
    m_Records[m_Count].resource.nickname = nickname;
    m_Records[m_Count].resource.requestType = RequestType_Faced;
    m_Records[m_Count].resource.requestStatus = RequestStatus_Unsent;
    m_Records[m_Count].resource.appInfo = appInfo;
    m_Records[m_Count].resource.inAppScreenName = inAppScreenName;
    m_Records[m_Count].resource.myInAppScreenName = myInAppScreenName;
    m_Count++;

    bool isSuccess = false;

    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            Invalidate();
        }
    };

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    // 新規にデータを作成する際、ストレージの容量不足にならないよう、不要なファイルを削除する。
    NN_RESULT_DO(AccountStorageManager::GetInstance().DeleteUnmanagedNetworkServiceAccountDirectory());
    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    NN_RESULT_DO(Save());
    NN_RESULT_DO(SaveProfileImage(data->accountId, image, imageSize));

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    isSuccess = true;

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::CancelFacedFriendRequest(const nn::account::Uid& uid,
    nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountId myAccountId;
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&myAccountId, uid));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int count = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, accountId,
        [](const Record& lhs, const nn::account::NetworkServiceAccountId& rhs)
        {
            return lhs.resource.accountId == rhs;
        });

    NN_RESULT_THROW_UNLESS(count < m_Count, ResultNetworkServiceAccountNotExistsGeneral());
    m_Count = count;

    bool isSuccess = false;

    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            Invalidate();
        }
    };

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    NN_RESULT_DO(Save());
    NN_RESULT_DO(DeleteProfileImage(accountId));

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    isSuccess = true;

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::GetFacedFriendRequestList(int* outCount, FriendRequestImpl* outRequests,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outRequests);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outRequests, 0, sizeof (FriendRequestImpl) * count);

    nn::account::NetworkServiceAccountId myAccountId;
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&myAccountId, uid));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = offset; i < m_Count; i++)
    {
        if (count-- == 0)
        {
            break;
        }

        FriendRequestImpl& impl = outRequests[actualCount++];

        impl.data.uid = uid;
        impl.data.accountId = m_Records[i].resource.accountId;
        impl.data.nickname = m_Records[i].resource.nickname;

        AccountStorageManager::MakeFacedFriendRequestProfileImageUrl(impl.data.profileImageUrl.value,
            sizeof (impl.data.profileImageUrl.value), uid, myAccountId, impl.data.accountId);

        impl.data.requestListType = RequestListType_Faced;
        impl.data.requestType = m_Records[i].resource.requestType;
        impl.data.requestStatus = m_Records[i].resource.requestStatus;
        impl.data.routeInfo.appInfo = m_Records[i].resource.appInfo;
        impl.data.routeInfo.name = m_Records[i].resource.inAppScreenName;
        impl.data.read = true;
        impl.data.isValid = true;
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::GetFacedFriendRequest(FacedFriendRequestResource* outResource,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outResource);

    std::memset(outResource, 0, sizeof (FacedFriendRequestResource));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    for (int i = 0; i < m_Count; i++)
    {
        if (m_Records[i].resource.accountId == accountId)
        {
            *outResource = m_Records[i].resource;
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultNetworkServiceAccountNotExistsGeneral());
}

nn::Result FacedFriendRequestManager::GetFacedFriendRequestList(int* outCount, FacedFriendRequestResource* outResources,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outResources);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outResources, 0, sizeof (FacedFriendRequestResource) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = offset; i < m_Count; i++)
    {
        if (count-- == 0)
        {
            break;
        }

        outResources[actualCount++] = m_Records[i].resource;
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::GetProfileImage(size_t* outSize,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    nn::account::NetworkServiceAccountId myAccountId;
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&myAccountId, uid));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    Url url;
    AccountStorageManager::MakeFacedFriendRequestProfileImageUrl(url.value, sizeof (url.value), uid, myAccountId, accountId);

    return GetProfileImage(outSize, url.value, buffer, size);
}

nn::Result FacedFriendRequestManager::GetProfileImage(size_t* outSize,
    const char* url, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(url);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    nn::account::Uid uid;
    char path[64];

    NN_RESULT_THROW_UNLESS(AccountStorageManager::ResolveFacedFriendRequestProfileImageUrl(&uid, path, sizeof (path), url),
        ResultProfileImageCacheNotFound());

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(outSize, handle, 0, buffer, size));

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::IsEmpty(bool* outIsEmpty, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsEmpty);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    *outIsEmpty = (m_Count == 0);

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::UpdateFacedFriendRequestList(const nn::account::Uid& uid,
    const FacedFriendRequestResource* resources, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(resources);
    NN_SDK_REQUIRES(count <= FacedFriendRequestCountMax);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    for (int i = 0; i < count; i++)
    {
        NN_RESULT_DO(UpdateFacedFriendRequest(resources[i]));
    }

    NN_RESULT_DO(Save());

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    NN_RESULT_SUCCESS;
}

void FacedFriendRequestManager::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_CurrentUid = nn::account::InvalidUid;
}

nn::Result FacedFriendRequestManager::Load(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadImpl(uid))
        NN_RESULT_CATCH_ALL
        {
            m_Count = 0;
        }
    NN_RESULT_END_TRY;

    m_CurrentUid = uid;

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT
{
    if (uid == m_CurrentUid)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    if ((size % sizeof (Record)) != 0)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. ((size % sizeof (Record)) != 0)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    int count = static_cast<int>(size / sizeof (Record));

    if (count > FacedFriendRequestCountMax)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (count > FacedFriendRequestCountMax)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (Record) * count));

    if (!Verify(count))
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (verification failed)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    m_Count = count;

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::Save() NN_NOEXCEPT
{
    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    if (m_Count == 0)
    {
        NN_RESULT_SUCCESS;
    }

    {
        NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (Record) * m_Count));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (Record) * m_Count,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_SUCCESS;
}

bool FacedFriendRequestManager::Verify(int count) NN_NOEXCEPT
{
    for (int i = 0; i < count; i++)
    {
        if (!(m_Records[i].resource.requestType == RequestType_Faced || m_Records[i].resource.requestType == RequestType_InApp))
        {
            return false;
        }
        if (!(m_Records[i].resource.requestStatus >= RequestStatus_Unsent &&
            m_Records[i].resource.requestStatus <= RequestStatus_UnsentForBlockedByMe))
        {
            return false;
        }
        if (!DataVerifier::VerifyString(m_Records[i].resource.nickname.name, sizeof (m_Records[i].resource.nickname.name)))
        {
            return false;
        }
    }

    return true;
}

nn::Result FacedFriendRequestManager::SaveProfileImage(nn::account::NetworkServiceAccountId accountId,
    const void* image, size_t imageSize) NN_NOEXCEPT
{
    char name[32] = {};
    nn::util::SNPrintf(name, sizeof (name), "faced/%016llx.jpg", accountId.id);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), name));

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(FileSystem::CreateFile(path, imageSize));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, image, imageSize,
        nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::DeleteProfileImage(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    char name[32] = {};
    nn::util::SNPrintf(name, sizeof (name), "faced/%016llx.jpg", accountId.id);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), name));

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result FacedFriendRequestManager::UpdateFacedFriendRequest(const FacedFriendRequestResource& resource) NN_NOEXCEPT
{
    int index = -1;

    for (int i = 0; i < m_Count; i++)
    {
        if (resource.accountId == m_Records[i].resource.accountId)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        NN_RESULT_SUCCESS;
    }

    if (resource.requestStatus == RequestStatus_Pending)
    {
        m_Count = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, resource.accountId,
            [](const Record& lhs, const nn::account::NetworkServiceAccountId& rhs)
            {
                return lhs.resource.accountId == rhs;
            });

        NN_RESULT_DO(DeleteProfileImage(resource.accountId));
    }
    else
    {
        m_Records[index].resource = resource;
    }

    NN_RESULT_SUCCESS;
}

}}}}}
