﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_EventWatcher.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskThread.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_BlockedUserListManager.h>
#include <nn/friends/detail/service/core/friends_FriendListManager.h>
#include <nn/friends/detail/service/core/friends_FriendPresenceManager.h>
#include <nn/friends/detail/service/core/friends_FriendRequestCountManager.h>
#include <nn/friends/detail/service/core/friends_FacedFriendRequestManager.h>
#include <nn/friends/detail/service/core/friends_PlayHistoryManager.h>
#include <nn/friends/detail/service/core/friends_PlayLogManager.h>
#include <nn/friends/detail/service/core/friends_UserPresenceManager.h>
#include <nn/friends/detail/service/core/friends_UserSettingManager.h>
#include <nn/friends/detail/service/core/friends_UuidManager.h>
#include <nn/friends/detail/service/util/friends_ApplicationInfo.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/npns/npns_Api.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[32 * 1024];

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);

    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();
    nn::os::SdkMutexType g_MutexForDaemon = NN_OS_SDK_MUTEX_INITIALIZER();

    nn::os::Event g_SuspendEvent(nn::os::EventClearMode_ManualClear);
    nn::os::Event g_SuspendedEvent(nn::os::EventClearMode_ManualClear);
    nn::os::Event g_ResumeEvent(nn::os::EventClearMode_ManualClear);

    int g_SuspendCount = 0;

    nn::account::Uid g_AllUsers[nn::account::UserCountMax];
    int g_UserCount = 0;

    nn::account::Uid g_OpenUsers[nn::account::UserCountMax];
    int g_OpenUserCount = 0;

    nn::account::Uid g_NsaAvailabilityUsers[nn::account::UserCountMax];
    int g_NsaAvailabilityUserCount = 0;
}

namespace
{
    void NotifyUserRegistrationChanged() NN_NOEXCEPT
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        int count;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, uids, NN_ARRAY_SIZE(uids)));

        // 削除されたユーザーアカウントの判定と通知を行う。
        for (int i = 0; i < g_UserCount; i++)
        {
            bool isChanged = true;

            for (int j = 0; j < count; j++)
            {
                if (g_AllUsers[i] == uids[j])
                {
                    isChanged = false;
                    break;
                }
            }

            if (isChanged)
            {
                // NN_DETAIL_FRIENDS_INFO("[friends] EventWatcher::NotifyUserRemoved(%016llx_%016llx)\n", g_AllUsers[i]._data[0], g_AllUsers[i]._data[1]);

                // ユーザーアカウントが削除された時、ユーザーアカウントに関するキャッシュを破棄する。
                // 同じ ID のユーザーアカウントが再生成されることはないため、ユーザーアカウント追加時のキャッシュ破棄は行わない。
                PlayHistoryManager::GetInstance().Invalidate();
                UuidManager::GetInstance().Invalidate();

                UserPresenceManager::GetInstance().NotifyUserRemoved(g_AllUsers[i]);
                PlayLogManager::GetInstance().NotifyUserRemoved(g_AllUsers[i]);
                FriendListManager::GetInstance().NotifyUserRemoved(g_AllUsers[i]);
                FriendPresenceManager::GetInstance().NotifyUserRemoved(g_AllUsers[i]);
                FriendRequestCountManager::GetInstance().NotifyUserRemoved(g_AllUsers[i]);

                BackgroundTaskManager::GetInstance().NotifyUserRemoved(g_AllUsers[i]);
            }
        }

        // 追加されたユーザーアカウントの判定と通知を行う。
        for (int i = 0; i < count; i++)
        {
            bool isChanged = true;

            for (int j = 0; j < g_UserCount; j++)
            {
                if (uids[i] == g_AllUsers[j])
                {
                    isChanged = false;
                    break;
                }
            }

            if (isChanged)
            {
                // NN_DETAIL_FRIENDS_INFO("[friends] EventWatcher::NotifyUserAdded(%016llx_%016llx)\n", uids[i]._data[0], uids[i]._data[1]);

                UserPresenceManager::GetInstance().NotifyUserAdded(uids[i]);
                PlayLogManager::GetInstance().NotifyUserAdded(uids[i]);
                FriendListManager::GetInstance().NotifyUserAdded(uids[i]);
                FriendPresenceManager::GetInstance().NotifyUserAdded(uids[i]);
                FriendRequestCountManager::GetInstance().NotifyUserAdded(uids[i]);

                BackgroundTaskManager::GetInstance().NotifyUserAdded(uids[i]);
            }
        }

        if (count > 0)
        {
            std::memcpy(g_AllUsers, uids, sizeof (nn::account::Uid) * count);
        }

        g_UserCount = count;
    }

    void NotifyUserStateChanged() NN_NOEXCEPT
    {
        // ユーザーアカウントの状態変化をトリガーにいっしょに遊んだユーザーの保存を試みる。
        PlayHistoryManager::GetInstance().Save();

        nn::account::Uid uids[nn::account::UserCountMax];
        int count;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListOpenUsers(&count, uids, NN_ARRAY_SIZE(uids)));

        // Close されたユーザーアカウントの判定と通知を行う。
        for (int i = 0; i < g_OpenUserCount; i++)
        {
            bool isChanged = true;

            for (int j = 0; j < count; j++)
            {
                if (g_OpenUsers[i] == uids[j])
                {
                    isChanged = false;
                    break;
                }
            }

            if (isChanged)
            {
                // NN_DETAIL_FRIENDS_INFO("[friends] EventWatcher::NotifyUserClosed(%016llx_%016llx)\n", g_OpenUsers[i]._data[0], g_OpenUsers[i]._data[1]);

                UserPresenceManager::GetInstance().NotifyUserClosed(g_OpenUsers[i]);
                PlayLogManager::GetInstance().NotifyUserClosed(g_OpenUsers[i]);

                BackgroundTaskManager::GetInstance().NotifyUserClosed(g_OpenUsers[i]);
            }
        }

        // Open されたユーザーアカウントの判定と通知を行う。
        for (int i = 0; i < count; i++)
        {
            bool isChanged = true;

            for (int j = 0; j < g_OpenUserCount; j++)
            {
                if (uids[i] == g_OpenUsers[j])
                {
                    isChanged = false;
                    break;
                }
            }

            if (isChanged)
            {
                // NN_DETAIL_FRIENDS_INFO("[friends] EventWatcher::NotifyUserOpened(%016llx_%016llx)\n", uids[i]._data[0], uids[i]._data[1]);

                ApplicationInfo appInfo = {};

                nn::Result result = detail::service::util::GetUserLastOpenedApplicationInfo(&appInfo, uids[i]);

                // アプリケーション情報が取得できなくても先に進む。
                // この場合、アプリケーション未指定でプレゼンスがオンライン状態になる。
                // プレイログに記録は残らない。
                if (result.IsFailure())
                {
                    NN_DETAIL_FRIENDS_INFO("[friends] GetUserLastOpenedApplicationInfo() failed. code = %03d-%04d\n",
                        result.GetModule(), result.GetDescription());
                }

                UserPresenceManager::GetInstance().NotifyUserOpened(uids[i], appInfo);
                PlayLogManager::GetInstance().NotifyUserOpened(uids[i], appInfo);

                BackgroundTaskManager::GetInstance().NotifyUserOpened(uids[i]);
            }
        }

        if (count > 0)
        {
            std::memcpy(g_OpenUsers, uids, sizeof (nn::account::Uid) * count);
        }

        g_OpenUserCount = count;
    }

    void NotifyNetworkServiceAccountAvailabilityChanged() NN_NOEXCEPT
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        int count = 0;

        for (int i = 0; i < g_UserCount; i++)
        {
            if (Account::IsNetworkServiceAccountAvailable(g_AllUsers[i]))
            {
                uids[count++] = g_AllUsers[i];
            }
        }

        // ネットワークサービスアカウントの紐付けが更新された時、ネットワークサービスアカウントに関するキャッシュを破棄する。
        UserSettingManager::GetInstance().Invalidate();
        PlayLogManager::GetInstance().Invalidate();
        FriendListManager::GetInstance().Invalidate();
        FacedFriendRequestManager::GetInstance().Invalidate();
        BlockedUserListManager::GetInstance().Invalidate();

        // ネットワークサービスアカウントが無効になったユーザーアカウントの判定と通知を行う。
        for (int i = 0; i < g_NsaAvailabilityUserCount; i++)
        {
            bool isChanged = true;

            for (int j = 0; j < count; j++)
            {
                if (g_NsaAvailabilityUsers[i] == uids[j])
                {
                    isChanged = false;
                    break;
                }
            }

            if (isChanged)
            {
                // NN_DETAIL_FRIENDS_INFO("[friends] EventWatcher::NotifyNetworkServiceAccountAvailabilityChanged[D](%016llx_%016llx)\n", g_NsaAvailabilityUsers[i]._data[0], g_NsaAvailabilityUsers[i]._data[1]);

                BackgroundTaskManager::GetInstance().NotifyNetworkServiceAccountAvailabilityChanged(g_NsaAvailabilityUsers[i], false);
                FriendListManager::GetInstance().NotifyNetworkServiceAccountAvailabilityChanged(g_NsaAvailabilityUsers[i], false);
                FriendRequestCountManager::GetInstance().NotifyNetworkServiceAccountAvailabilityChanged(g_NsaAvailabilityUsers[i], false);
            }
        }

        // ネットワークサービスアカウントが有効になったユーザーアカウントの判定と通知を行う。
        for (int i = 0; i < count; i++)
        {
            bool isChanged = true;

            for (int j = 0; j < g_NsaAvailabilityUserCount; j++)
            {
                if (uids[i] == g_NsaAvailabilityUsers[j])
                {
                    isChanged = false;
                    break;
                }
            }

            if (isChanged)
            {
                // NN_DETAIL_FRIENDS_INFO("[friends] EventWatcher::NotifyNetworkServiceAccountAvailabilityChanged[E](%016llx_%016llx)\n", uids[i]._data[0], uids[i]._data[1]);

                BackgroundTaskManager::GetInstance().NotifyNetworkServiceAccountAvailabilityChanged(uids[i], true);
                FriendListManager::GetInstance().NotifyNetworkServiceAccountAvailabilityChanged(uids[i], true);
                FriendRequestCountManager::GetInstance().NotifyNetworkServiceAccountAvailabilityChanged(uids[i], true);
            }
        }

        if (count > 0)
        {
            std::memcpy(g_NsaAvailabilityUsers, uids, sizeof (nn::account::Uid) * count);
        }

        g_NsaAvailabilityUserCount = count;
    }

    void NotifyNpnsStateChanged() NN_NOEXCEPT
    {
        nn::npns::State state = nn::npns::GetState();

        // NN_DETAIL_FRIENDS_INFO("[friends] Npns state was changed (%d).\n", state);

        bool isConnected = (state == nn::npns::State_Connected);

        UserPresenceManager::GetInstance().NotifyNpnsStateChanged(isConnected);
        FriendPresenceManager::GetInstance().NotifyNpnsStateChanged(isConnected);
        BackgroundTaskManager::GetInstance().NotifyNpnsStateChanged(isConnected);
    }

    void WaitWhileDaemonIsSuspended() NN_NOEXCEPT
    {
        bool isSuspendRequired = false;
        {
            std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

            if (g_SuspendCount > 0)
            {
                isSuspendRequired = true;
            }
        }
        if (isSuspendRequired)
        {
            g_SuspendedEvent.Signal();

            nn::os::WaitAny(g_StopEvent.GetBase(), g_ResumeEvent.GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }

            g_ResumeEvent.Clear();

            g_SuspendedEvent.Clear();
        }
    }

    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::account::Notifier accountRegistrationNotifier;
        nn::os::SystemEvent accountRegistrationEvent;

        nn::account::Notifier accountStateChangeNotifier;
        nn::os::SystemEvent accountStateChangeEvent;

        nn::account::Notifier accountNsaAvailabilityChangeNotifier;
        nn::os::SystemEvent accountNsaAvailabilityChangeEvent;

        nn::os::SystemEvent npnsStateChangeEvent;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserRegistrationNotifier(&accountRegistrationNotifier));
        NN_ABORT_UNLESS_RESULT_SUCCESS(accountRegistrationNotifier.GetSystemEvent(&accountRegistrationEvent));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserStateChangeNotifier(&accountStateChangeNotifier));
        NN_ABORT_UNLESS_RESULT_SUCCESS(accountStateChangeNotifier.GetSystemEvent(&accountStateChangeEvent));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAvailabilityChangeNotifier(&accountNsaAvailabilityChangeNotifier));
        NN_ABORT_UNLESS_RESULT_SUCCESS(accountNsaAvailabilityChangeNotifier.GetSystemEvent(&accountNsaAvailabilityChangeEvent));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::GetStateChangeEvent(npnsStateChangeEvent));

        while (NN_STATIC_CONDITION(true))
        {
            WaitWhileDaemonIsSuspended();

            nn::os::WaitAny(g_StopEvent.GetBase(),
                g_SuspendEvent.GetBase(),
                accountRegistrationEvent.GetBase(),
                accountStateChangeEvent.GetBase(),
                accountNsaAvailabilityChangeEvent.GetBase(),
                npnsStateChangeEvent.GetBase());

            if (g_StopEvent.TryWait())
            {
                break;
            }
            if (g_SuspendEvent.TryWait())
            {
                continue;
            }

            if (accountRegistrationEvent.TryWait())
            {
                accountRegistrationEvent.Clear();
                NotifyUserRegistrationChanged();
            }
            if (accountStateChangeEvent.TryWait())
            {
                accountStateChangeEvent.Clear();
                NotifyUserStateChanged();
            }
            if (accountNsaAvailabilityChangeEvent.TryWait())
            {
                accountNsaAvailabilityChangeEvent.Clear();
                NotifyNetworkServiceAccountAvailabilityChanged();
            }
            if (npnsStateChangeEvent.TryWait())
            {
                npnsStateChangeEvent.Clear();
                NotifyNpnsStateChanged();
            }
        }
    }
}

void EventWatcher::Iniaialize() NN_NOEXCEPT
{
    NotifyUserRegistrationChanged();
    NotifyNetworkServiceAccountAvailabilityChanged();
}

void EventWatcher::Start() NN_NOEXCEPT
{
    g_StopEvent.Clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(friends, EventWatcher)));

    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(friends, EventWatcher));
    nn::os::StartThread(&g_Thread);
}

void EventWatcher::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::DestroyThread(&g_Thread);
}

void EventWatcher::Suspend() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_MutexForDaemon)> lockForDaemon(g_MutexForDaemon);

    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        g_SuspendCount++;
        g_SuspendEvent.Signal();
    }

    g_SuspendedEvent.Wait();
}

void EventWatcher::Resume() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_MutexForDaemon)> lockForDaemon(g_MutexForDaemon);
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    if (g_SuspendCount == 0)
    {
        return;
    }
    if (--g_SuspendCount == 0)
    {
        g_SuspendEvent.Clear();
        g_ResumeEvent.Signal();
    }
}

}}}}}
