﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_DataVerifier.h>
#include <nn/friends/detail/util/friends_LanguageCode.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

bool DataVerifier::VerifyString(const char* string, size_t maxLength) NN_NOEXCEPT
{
    size_t length = static_cast<size_t>(nn::util::Strnlen(string, static_cast<int>(maxLength)));

    if (length == 0)
    {
        return true;
    }
    if (length == maxLength)
    {
        return false;
    }

    return nn::util::VerifyUtf8String(string, length);
}

bool DataVerifier::VerifyString(const char* string, size_t maxLength, int maxCount) NN_NOEXCEPT
{
    size_t length = static_cast<size_t>(nn::util::Strnlen(string, static_cast<int>(maxLength)));

    if (length == 0)
    {
        return true;
    }
    if (length == maxLength)
    {
        return false;
    }

    int count = nn::util::GetCodePointCountOfUtf8String(string, length);

    if (count == -1 || count > maxCount)
    {
        return false;
    }

    return true;
}

bool DataVerifier::VerifyInAppScreenName(const InAppScreenName& name) NN_NOEXCEPT
{
    if (!VerifyString(name.name, sizeof (name.name), InAppScreenNameLengthMax))
    {
        return false;
    }
    if (!detail::util::VerifyLanguageCode(name.language))
    {
        return false;
    }

    return true;
}

bool DataVerifier::VerifyMiiImageUrlParam(const MiiImageUrlParam& param) NN_NOEXCEPT
{
    int length = nn::util::Strnlen(param.value, sizeof (param.value));

    if (length == 0 || length == sizeof (param.value))
    {
        return false;
    }

    for (int i = 0; i < length; i++)
    {
        char c = param.value[i];

        // 36 進数の範囲内であることを確認する。
        if (!((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z')))
        {
            return false;
        }
    }

    return true;
}

bool DataVerifier::VerifyPosixTime(const nn::time::PosixTime& posixTime) NN_NOEXCEPT
{
    // 9999-12-31 23:59:59 のエポック秒。
    const int64_t DateTimeMax = 253402300799ll;

    return (posixTime.value >= 0 && posixTime.value <= DateTimeMax);
}

bool DataVerifier::VerifyFriendCode(const FriendCode& friendCode) NN_NOEXCEPT
{
    int length = nn::util::Strnlen(friendCode.value, sizeof (friendCode.value));

    if (length == sizeof (friendCode.value))
    {
        return false;
    }

    // フレンドコード未発効の場合があるので、長さ 0 は許容する。
    for (int i = 0; i < length; i++)
    {
        char c = friendCode.value[i];

        // 表示可能な ASCII コードの範囲内かどうかを確認する。
        if (c < 0x20 || c > 0x7E)
        {
            return false;
        }
    }

    return true;
}

bool DataVerifier::VerifyPlayLog(const PlayLog& playLog) NN_NOEXCEPT
{
    if (playLog.appInfo.appId.value == 0)
    {
        return (playLog.appInfo.presenceGroupId == 0 &&
            playLog.totalPlayCount == 0 && playLog.totalPlayTime == 0 &&
            playLog.firstPlayedTime.value == 0 && playLog.lastPlayedTime.value == 0);
    }

    return (playLog.totalPlayCount > 0 && playLog.totalPlayTime > 0 &&
        VerifyPosixTime(playLog.firstPlayedTime) && VerifyPosixTime(playLog.lastPlayedTime));
}

bool DataVerifier::VerifyUserSetting(const UserSettingImpl& setting) NN_NOEXCEPT
{
    if (setting.data.uid == nn::account::InvalidUid)
    {
        return false;
    }
    if (setting.data.presencePermission < nn::friends::PresencePermission_Self ||
        setting.data.presencePermission > nn::friends::PresencePermission_Friends)
    {
        return false;
    }
    if (setting.data.playLogPermission < nn::friends::ExtraPermission_Self ||
        setting.data.playLogPermission > nn::friends::ExtraPermission_Everyone)
    {
        return false;
    }
    if (!VerifyFriendCode(setting.data.friendCode))
    {
        return false;
    }
    if (!VerifyPosixTime(setting.data.friendCodeNextIssuableTime))
    {
        return false;
    }

    for (int i = 0; i < NN_ARRAY_SIZE(setting.data.profileExtra.data.playLog); i++)
    {
        if (!VerifyPlayLog(setting.data.profileExtra.data.playLog[i]))
        {
            return false;
        }
    }

    return true;
}

}}}}}
