﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_BackgroundTaskThread.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_UserPresenceManager.h>
#include <nn/friends/detail/service/core/task/friends_BgTaskGetListSummary.h>
#include <nn/friends/detail/service/core/task/friends_TaskSyncUserSetting.h>
#include <nn/friends/detail/service/core/task/friends_TaskSendFacedFriendRequest.h>
#include <nn/friends/detail/service/core/task/friends_TaskSyncFriendList.h>
#include <nn/friends/detail/service/core/task/friends_TaskSyncBlockedUserList.h>
#include <nn/friends/detail/service/core/task/friends_BgTaskDownloadImage.h>
#include <nn/friends/detail/service/core/task/friends_BgTaskUpdateUserPresence.h>
#include <nn/friends/detail/service/core/task/friends_BgTaskSyncFriendPresence.h>
#include <nn/friends/detail/service/core/task/friends_BgTaskGetFriendRequestCount.h>

#define TASK_BUFFER_SIZE 1536 // NOLINT(preprocessor/const)

#define CREATE_TASK(TCLASS, uid) \
    do                                                          \
    {                                                           \
        NN_STATIC_ASSERT(sizeof (TCLASS) <= TASK_BUFFER_SIZE);  \
                                                                \
        TCLASS* p = new (buffer) TCLASS;                        \
        p->SetUid(uid);                                         \
        return p;                                               \
    }                                                           \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[48 * 1024];

    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();
    nn::os::SdkMutexType g_MutexForDaemon = NN_OS_SDK_MUTEX_INITIALIZER();

    Task* g_RunningTask = nullptr;

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);

    nn::os::Event g_SuspendEvent(nn::os::EventClearMode_ManualClear);
    nn::os::Event g_SuspendedEvent(nn::os::EventClearMode_ManualClear);
    nn::os::Event g_ResumeEvent(nn::os::EventClearMode_ManualClear);

    int g_SuspendCount = 0;
}

namespace
{
    Task* CreateBackgroundTask(void* buffer, const BackgroundTaskManager::TaskInfo& info) NN_NOEXCEPT
    {
        switch (info.taskId)
        {
        case BackgroundTaskManager::TaskId_GetListSummary:
            {
                CREATE_TASK(GetListSummaryBgTask, info.uid);
            }
        case BackgroundTaskManager::TaskId_SyncUser:
            {
                CREATE_TASK(SyncUserSettingTask, info.uid);
            }
        case BackgroundTaskManager::TaskId_SyncFriendList:
            {
                CREATE_TASK(SyncFriendListTask, info.uid);
            }
        case BackgroundTaskManager::TaskId_SyncBlockedUserList:
            {
                CREATE_TASK(SyncBlockedUserListTask, info.uid);
            }
        case BackgroundTaskManager::TaskId_SendFacedFriendRequest:
            {
                CREATE_TASK(SendFacedFriendRequestTask, info.uid);
            }
        case BackgroundTaskManager::TaskId_DownloadFriendProfileImage:
            {
                CREATE_TASK(DownloadImageBgTask, info.uid);
            }
        case BackgroundTaskManager::TaskId_UpdateUserPresence:
            {
                CREATE_TASK(UpdateUserPresenceBgTask, info.uid);
            }
        case BackgroundTaskManager::TaskId_GetFriendPresence:
            {
                CREATE_TASK(SyncFriendPresenceBgTask, info.uid);
            }
        case BackgroundTaskManager::TaskId_GetFriendRequestCount:
            {
                CREATE_TASK(GetFriendRequestCountBgTask, info.uid);
            }
        default:
            NN_ABORT("");
        }
    }

#if defined (NN_DETAIL_ENABLE_SDK_STRUCTURED_LOG)

    const char* GetTaskNameString(BackgroundTaskManager::TaskId id) NN_NOEXCEPT
    {
        switch (id)
        {
        case BackgroundTaskManager::TaskId_GetListSummary:
            return "GetListSummary";
        case BackgroundTaskManager::TaskId_SyncUser:
            return "SyncUser";
        case BackgroundTaskManager::TaskId_SyncFriendList:
            return "SyncFriendList";
        case BackgroundTaskManager::TaskId_SyncBlockedUserList:
            return "SyncBlockedUserList";
        case BackgroundTaskManager::TaskId_SendFacedFriendRequest:
            return "SendFacedFriendRequest";
        case BackgroundTaskManager::TaskId_DownloadFriendProfileImage:
            return "DownloadFriendProfileImage";
        case BackgroundTaskManager::TaskId_UpdateUserPresence:
            return "UpdateUserPresence";
        case BackgroundTaskManager::TaskId_GetFriendPresence:
            return "GetFriendPresence";
        case BackgroundTaskManager::TaskId_GetFriendRequestCount:
            return "GetFriendRequestCount";
        default:
            return "-";
        }
    }

#endif

    nn::Result RunTask(const BackgroundTaskManager::TaskInfo& info) NN_NOEXCEPT
    {
        Bit64 buffer[TASK_BUFFER_SIZE / sizeof (Bit64)];

        Task* p = CreateBackgroundTask(buffer, info);

        NN_UTIL_SCOPE_EXIT
        {
            p->~Task();
        };

        {
            std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);
            g_RunningTask = p;
        }

        p->Run();
        p->Wait();

        {
            std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);
            g_RunningTask = nullptr;
        }

        return p->GetResult();
    }

    nn::Result Process(BackgroundTaskManager::TaskInfo& info) NN_NOEXCEPT
    {
        // 有効なネットワークサービスアカウントを保持していない場合、何もしない。
        NN_RESULT_THROW_UNLESS(Account::IsNetworkServiceAccountAvailable(info.uid), ResultNetworkServiceAccountNotLinked());

        nn::nifm::NetworkConnection connection;

        NN_DETAIL_FRIENDS_INFO("[friends] I wait until establishment of the network connection...\n");

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(connection.GetRequestHandle(),
           nn::nifm::RequirementPreset_InternetForSystemProcessContinuous));

        connection.SubmitRequest();

        do
        {
            nn::os::WaitAny(g_StopEvent.GetBase(), g_SuspendEvent.GetBase(), connection.GetSystemEvent().GetBase());

            if (g_StopEvent.TryWait() || g_SuspendEvent.TryWait())
            {
                NN_RESULT_THROW(ResultSuspended());
            }

            connection.GetSystemEvent().Clear();
        }
        while (!connection.IsAvailable());

        NN_DETAIL_FRIENDS_INFO("[friends] The network connection was established!\n");

        nn::Result result = RunTask(info);

        // 処理中にネットワークから切断された場合、未接続エラー扱いとする。（スリープによる切断・再接続は SystemEvent の Signal で検出）
        if (result.IsFailure() && (!connection.IsAvailable() || connection.GetSystemEvent().TryWait()))
        {
            NN_DETAIL_FRIENDS_INFO("[friends] The network disconnection was detected. code = %03d-%04d -> %03d-%04d\n",
                result.GetModule(), result.GetDescription(),
                ResultInternetRequestNotAccepted().GetModule(), ResultInternetRequestNotAccepted().GetDescription());

            NN_RESULT_THROW(ResultInternetRequestNotAccepted());
        }

        return result;
    }

    void WaitWhileDaemonIsSuspended() NN_NOEXCEPT
    {
        bool isSuspendRequired = false;
        {
            std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

            if (g_SuspendCount > 0)
            {
                isSuspendRequired = true;
            }
        }
        if (isSuspendRequired)
        {
            g_SuspendedEvent.Signal();

            nn::os::WaitAny(g_StopEvent.GetBase(), g_ResumeEvent.GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }

            g_ResumeEvent.Clear();

            g_SuspendedEvent.Clear();
        }
    }

    void StartUserPresencePeriodicUpdate() NN_NOEXCEPT
    {
        nn::account::Uid uids[nn::account::UserCountMax];
        int count;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, uids, NN_ARRAY_SIZE(uids)));

        for (int i = 0; i < count; i++)
        {
            BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(uids[i]);
        }
    }

    void WorkerThread(void*) NN_NOEXCEPT
    {
        StartUserPresencePeriodicUpdate();

        while (NN_STATIC_CONDITION(true))
        {
            WaitWhileDaemonIsSuspended();

            nn::os::WaitAny(g_StopEvent.GetBase(),
                g_SuspendEvent.GetBase(),
                BackgroundTaskManager::GetInstance().GetScheduleEvent().GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }
            if (g_SuspendEvent.TryWait())
            {
                continue;
            }

            if (BackgroundTaskManager::GetInstance().GetScheduleEvent().TryWait())
            {
                NN_DETAIL_FRIENDS_INFO("[friends] (!) BackgroundTaskManager::ScheduleEvent was signaled.\n");
                BackgroundTaskManager::GetInstance().GetScheduleEvent().Clear();

                BackgroundTaskManager::GetInstance().Schedule();
            }

            BackgroundTaskManager::TaskInfo info = {};

            while (BackgroundTaskManager::GetInstance().GetRunnableTask(&info))
            {
                NN_DETAIL_FRIENDS_INFO("[friends] (@) Task(%s, %016llx_%016llx)...\n",
                    GetTaskNameString(info.taskId), info.uid._data[0], info.uid._data[1]);

                nn::Result result = Process(info);

                if (result.IsSuccess())
                {
                    NN_DETAIL_FRIENDS_INFO("[friends] (@) Task(%s, %016llx_%016llx) done!\n",
                        GetTaskNameString(info.taskId), info.uid._data[0], info.uid._data[1]);
                }
                else
                {
                    NN_DETAIL_FRIENDS_INFO("[friends] (@) Task(%s, %016llx_%016llx) done! code = %03d-%04d\n",
                        GetTaskNameString(info.taskId), info.uid._data[0], info.uid._data[1],
                        result.GetModule(), result.GetDescription());
                }

                BackgroundTaskManager::GetInstance().NotifyDone(info, result);

                if (ResultSuspended::Includes(result))
                {
                    break;
                }
            }
        }
    }
}

void BackgroundTaskThread::Start() NN_NOEXCEPT
{
    g_StopEvent.Clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(friends, BackgroundTask)));

    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(friends, BackgroundTask));
    nn::os::StartThread(&g_Thread);
}

void BackgroundTaskThread::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::DestroyThread(&g_Thread);
}

void BackgroundTaskThread::Suspend() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_MutexForDaemon)> lockForDaemon(g_MutexForDaemon);

    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        g_SuspendCount++;
        g_SuspendEvent.Signal();

        if (g_RunningTask)
        {
            NN_DETAIL_FRIENDS_INFO("[friends] Cancel running task. (suspend)\n");
            g_RunningTask->Cancel();
        }
    }

    g_SuspendedEvent.Wait();
}

void BackgroundTaskThread::Resume() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_MutexForDaemon)> lockForDaemon(g_MutexForDaemon);
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    if (g_SuspendCount == 0)
    {
        return;
    }
    if (--g_SuspendCount == 0)
    {
        g_SuspendEvent.Clear();
        g_ResumeEvent.Signal();
    }
}

void BackgroundTaskThread::NotifySleep() NN_NOEXCEPT
{
}

void BackgroundTaskThread::NotifySleepAwaked() NN_NOEXCEPT
{
}

void BackgroundTaskThread::NotifyMinimumAwaked() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    // ソケットを作成する直前にスリープしていた場合、別モジュールが BG 処理をしていると通信が成功してしまう可能性がある。
    if (g_RunningTask)
    {
        NN_DETAIL_FRIENDS_INFO("[friends] Cancel running task. (minimum awaked)\n");
        g_RunningTask->Cancel();
    }
}

}}}}}
