﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_FacedFriendRequestManager.h>
#include <nn/account/http/account_ResultForHttp.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    // 再試行の時間間隔（30 秒 ⇒ 5 分 ⇒ 10 分 ⇒ 20 分 ⇒ 30 分 ⇒ 以降、30 分）
    const int RetryIntervalTable[] =
    {
        0, 30, 300, 600, 1200, 1800
    };
}

namespace
{
    bool IsCommunicationError(nn::Result result) NN_NOEXCEPT
    {
        return (ResultHttpError::Includes(result) || nn::account::http::ResultCurlError::Includes(result));
    }

    bool IsServerTemporaryError(nn::Result result) NN_NOEXCEPT
    {
        return (ResultServerError500InternalServerError::Includes(result) || ResultUnexpectedServerError::Includes(result));
    }
}

BackgroundTaskManager::BackgroundTaskManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_ScheduleEvent(nn::os::EventClearMode_ManualClear),
    m_IsConnected(false),
    m_IsSleeping(false),
    m_IsSuspended(false),
    m_RetryIntervalTableIndex(0)
{
    std::memset(m_Records, 0, sizeof (m_Records));

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        m_Records[i].uid = nn::account::InvalidUid;
    }
}

void BackgroundTaskManager::Schedule() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::os::Tick currentTick = nn::os::GetSystemTick();

    if (m_IsSuspended)
    {
        if (m_ResumeTick <= currentTick)
        {
            m_IsSuspended = false;
        }
    }

    if (!m_IsSuspended)
    {
        for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
        {
            if (m_Records[i].uid == nn::account::InvalidUid)
            {
                continue;
            }

            for (int n = 0; n < NN_ARRAY_SIZE(m_Records[i].tasks); n++)
            {
                if (m_Records[i].tasks[n].status == TaskStatus_Wait && m_Records[i].tasks[n].nextRunnableTick <= currentTick)
                {
                    if (n == TaskId_UpdateUserPresence && RescheduleUpdateUserPresenceTask(i))
                    {
                        continue;
                    }

                    SetRunnableWithoutSignal(i, static_cast<TaskId>(n));
                }
            }
        }
    }

    SetScheduleTimer();
}

bool BackgroundTaskManager::GetRunnableTask(TaskInfo* outInfo) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInfo);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_IsSleeping)
    {
        return false;
    }
    if (m_IsSuspended)
    {
        return false;
    }

    int currentPriority = 0;
    bool isFound = false;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == nn::account::InvalidUid)
        {
            continue;
        }

        for (int n = 0; n < NN_ARRAY_SIZE(m_Records[i].tasks); n++)
        {
            if (m_Records[i].tasks[n].status == TaskStatus_Runnable)
            {
                int priority = GetPriority(static_cast<TaskId>(n));

                if (priority > currentPriority)
                {
                    outInfo->taskId = static_cast<TaskId>(n);
                    outInfo->uid = m_Records[i].uid;
                    outInfo->revision = m_Records[i].tasks[n].revision;

                    currentPriority = priority;
                }

                isFound = true;
            }
        }
    }

    return isFound;
}

nn::os::TimerEvent& BackgroundTaskManager::GetScheduleEvent() NN_NOEXCEPT
{
    return m_ScheduleEvent;
}

void BackgroundTaskManager::NotifyDone(const TaskInfo& info, nn::Result result) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(info.uid);

    if (index == -1)
    {
        return;
    }

    if (result.IsSuccess())
    {
        m_Records[index].tasks[info.taskId].lastRunTick = nn::os::GetSystemTick();

        if (info.revision == m_Records[index].tasks[info.taskId].revision)
        {
            if (info.taskId == TaskId_UpdateUserPresence)
            {
                Wait(index, TaskId_UpdateUserPresence, PresenceKeepAliveInterval);
            }
            else
            {
                m_Records[index].tasks[info.taskId].status = TaskStatus_Done;
            }
        }
        else
        {
            NN_DETAIL_FRIENDS_INFO("[friends] The task revision was updated. (%lld -> %lld)\n",
                info.revision, m_Records[index].tasks[info.taskId].revision);

            if (info.taskId == TaskId_UpdateUserPresence)
            {
                Wait(index, TaskId_UpdateUserPresence, PresenceUpdateInterval);
            }
            else
            {
                if (m_Records[index].tasks[info.taskId].status == TaskStatus_Wait)
                {
                    // 次回実行時間を更新せずに再スケジュールする。
                    SetScheduleTimer();
                }
            }
        }

        // 通信が成功したので、リトライ間隔をリセットする。
        m_RetryIntervalTableIndex = 0;
    }
    else if (ResultInternetRequestNotAccepted::Includes(result))
    {
        // ネットワーク接続が有効ではなくなったため、有効になった後再試行する。
        Wait(index, info.taskId);
    }
    else if (ResultCanceled::Includes(result) || ResultSuspended::Includes(result))
    {
        // Suspend によるキャンセルになるため、Resume 後に動作開始する。
        Wait(index, info.taskId);
    }
    else if (IsCommunicationError(result) || IsServerTemporaryError(result))
    {
        if (m_RetryIntervalTableIndex < NN_ARRAY_SIZE(RetryIntervalTable) - 1)
        {
            m_RetryIntervalTableIndex++;
        }

        NN_DETAIL_FRIENDS_INFO("[friends] Retry... (wait = %d)\n", RetryIntervalTable[m_RetryIntervalTableIndex]);

        // 通信エラーが発生した場合、サーバーが過負荷状態になっている可能性がある。
        // そのため、一定期間サーバーへのアクセスを自重する。
        SuspendTemporarily(RetryIntervalTable[m_RetryIntervalTableIndex]);

        // 一定期間サーバーへのアクセスを自重した後、再試行する。
        Wait(index, info.taskId);
    }
    else
    {
        // 再試行による復帰が見込めないため、完了扱いとする。
        m_Records[index].tasks[info.taskId].status = TaskStatus_Done;
    }
}

void BackgroundTaskManager::NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = -1;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == nn::account::InvalidUid)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        return;
    }

    std::memset(&m_Records[index], 0, sizeof (Record));
    m_Records[index].uid = uid;

    bool isEmpty = false;

    // 対面フレンド申請の確認を行う。
    if (FacedFriendRequestManager::GetInstance().IsEmpty(&isEmpty, uid).IsSuccess() && !isEmpty)
    {
        NotifyFacedFriendRequestAdded(uid);
    }
}

void BackgroundTaskManager::NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    m_Records[index].uid = nn::account::InvalidUid;
}

void BackgroundTaskManager::NotifyUserOpened(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    // ユーザーアカウントの Open を即時に通知するため、タスクの実行を一時停止していたら即時再開する。
    Resume();

    // TODO: プレゼンスの状態更新タスクをここで発行する？

    // フレンドリストの同期はサマリー情報取得時に行われるため、フレンドリスト同期のタスクは発行しない。
    if (m_IsConnected)
    {
        Wait(index, TaskId_GetListSummary);
    }
    else
    {
        m_Records[index].tasks[TaskId_GetListSummary].status = TaskStatus_WaitForOnline;
    }

    Wait(index, TaskId_SyncUser);
    Wait(index, TaskId_SyncBlockedUserList);
}

void BackgroundTaskManager::NotifyUserClosed(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    // ユーザーアカウントの Close を即時に通知するため、タスクの実行を一時停止していたら即時再開する。
    Resume();

    Wait(index, TaskId_SyncUser);

    // ユーザーアカウントを Close したため、サマリー情報の取得は取り下げる。
    m_Records[index].tasks[TaskId_GetListSummary].status = TaskStatus_Done;
}

void BackgroundTaskManager::NotifyNetworkServiceAccountAvailabilityChanged(const nn::account::Uid& uid, bool isAvailable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!isAvailable)
    {
        return;
    }

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    Wait(index, TaskId_SyncUser);
    Wait(index, TaskId_SyncFriendList);
    Wait(index, TaskId_SyncBlockedUserList);
    Wait(index, TaskId_UpdateUserPresence);
}

void BackgroundTaskManager::NotifyUserPresenceUpdated(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    int64_t elapsedTime = (nn::os::GetSystemTick() - m_Records[index].tasks[TaskId_UpdateUserPresence].lastRunTick).ToTimeSpan().GetSeconds();

    // API を連続で呼び出して更新が行われた時、できるだけ 1 回の更新処理で済むよう少しだけ待機する。
    Wait(index, TaskId_UpdateUserPresence,
        static_cast<int>(elapsedTime >= PresenceUpdateInterval ? 1 : PresenceUpdateInterval - elapsedTime));
}

void BackgroundTaskManager::NotifyFriendRequestReceived(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    Wait(index, TaskId_GetFriendRequestCount);
}

void BackgroundTaskManager::NotifyFriendRequestAccepted(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    Wait(index, TaskId_SyncFriendList);
    Wait(index, TaskId_GetFriendRequestCount);
}

void BackgroundTaskManager::NotifyFriendDeleted(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    Wait(index, TaskId_SyncFriendList);
}

void BackgroundTaskManager::NotifyFacedFriendRequestAdded(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    Wait(index, TaskId_SendFacedFriendRequest);
}

void BackgroundTaskManager::RequestSyncFriendList(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    // 即時実行要求が行われたため、タスクの実行を一時停止していたら即時再開する。
    Resume();

    // リスト情報のサマリーを取得する時にフレンドリストの同期を行うため、このタスクが予約されていたら何もしない。
    if (m_Records[index].tasks[TaskId_GetListSummary].status != TaskStatus_Wait)
    {
        Wait(index, TaskId_SyncFriendList);
    }
}

void BackgroundTaskManager::RequestUpdateFriendProfileImageCache(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = FindUser(uid);

    if (index == -1)
    {
        return;
    }

    Wait(index, TaskId_DownloadFriendProfileImage);
}

void BackgroundTaskManager::RequestUpdateListSummary() NN_NOEXCEPT
{
    nn::account::Uid uids[nn::account::UserCountMax];
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListOpenUsers(&count, uids, NN_ARRAY_SIZE(uids)));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < count; i++)
    {
        int index = FindUser(uids[i]);

        if (index != -1)
        {
            if (m_IsConnected)
            {
                Wait(index, TaskId_GetListSummary);
            }
            else
            {
                m_Records[index].tasks[TaskId_GetListSummary].status = TaskStatus_WaitForOnline;
            }
        }
    }
}

void BackgroundTaskManager::NotifySleep() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_IsSleeping = true;
}

void BackgroundTaskManager::NotifySleepAwaked() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_SleepAwakedTick = nn::os::GetSystemTick();

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == nn::account::InvalidUid)
        {
            continue;
        }

        Wait(i, TaskId_SyncUser);
        Wait(i, TaskId_SyncFriendList);
        Wait(i, TaskId_SyncBlockedUserList);

        if (m_Records[i].tasks[TaskId_UpdateUserPresence].status == TaskStatus_Wait ||
            m_Records[i].tasks[TaskId_UpdateUserPresence].status == TaskStatus_Runnable)
        {
            RescheduleUpdateUserPresenceTask(i);
        }
    }

    m_IsSleeping = false;
}

void BackgroundTaskManager::NotifyNpnsStateChanged(bool isConnected) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (isConnected == m_IsConnected)
    {
        return;
    }

    m_IsConnected = isConnected;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == nn::account::InvalidUid)
        {
            continue;
        }

        if (isConnected)
        {
            // NPNS サーバーが障害から復帰した時、多数台が一斉に接続することで間接的に BaaS に負荷が掛からないようにするためランダムウェイトを入れる。
            int randomWait = static_cast<int>(detail::service::util::GetRandom() % (RandomWaitMax - RandomWaitMin + 1)) + RandomWaitMin;

            NN_DETAIL_FRIENDS_INFO("[friends] Random wait... (%d seconds)\n", randomWait);

            Wait(i, TaskId_UpdateUserPresence, randomWait);
            Wait(i, TaskId_GetFriendPresence, randomWait);

            // リスト情報のサマリーを取得する時にフレンド申請の受信件数を取得するため、このタスクが予約・実行されていたら何もしない。
            if (m_Records[i].tasks[TaskId_GetListSummary].status == TaskStatus_Done)
            {
                Wait(i, TaskId_GetFriendRequestCount, randomWait);
            }

            if (m_Records[i].tasks[TaskId_GetListSummary].status == TaskStatus_WaitForOnline)
            {
                Wait(i, TaskId_GetListSummary, randomWait);
            }
        }
        else
        {
            if (m_Records[i].tasks[TaskId_GetListSummary].status == TaskStatus_Wait)
            {
                m_Records[i].tasks[TaskId_GetListSummary].status = TaskStatus_WaitForOnline;
            }
        }
    }
}

void BackgroundTaskManager::NotifyForegroundCommunicationDone(nn::Result result) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    // 既知のサーバーエラー（テンポラリエラーを除く）はサーバーとの通信が正常に行われたとみなす。
    if (result.IsSuccess() || (ResultServerError::Includes(result) && !IsServerTemporaryError(result)))
    {
        Resume();
    }
}

int BackgroundTaskManager::FindUser(const nn::account::Uid& uid) const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == uid)
        {
            return i;
        }
    }

    return -1;
}

void BackgroundTaskManager::SetRunnable(int index, TaskId taskId) NN_NOEXCEPT
{
    m_Records[index].tasks[taskId].status = TaskStatus_Runnable;

    if (m_IsSuspended)
    {
        nn::TimeSpan interval = (m_ResumeTick - nn::os::GetSystemTick()).ToTimeSpan();

        if (interval > 0)
        {
            m_ScheduleEvent.StartOneShot(interval);
        }
        else
        {
            m_ScheduleEvent.Signal();
        }
    }
    else
    {
        m_ScheduleEvent.Signal();
    }
}

void BackgroundTaskManager::SetRunnableWithoutSignal(int index, TaskId taskId) NN_NOEXCEPT
{
    m_Records[index].tasks[taskId].status = TaskStatus_Runnable;
}

void BackgroundTaskManager::Wait(int index, TaskId taskId, int wait) NN_NOEXCEPT
{
    m_Records[index].tasks[taskId].revision++;

    if (taskId == TaskId_UpdateUserPresence)
    {
        int64_t elapsedTime = (nn::os::GetSystemTick() - m_SleepAwakedTick).ToTimeSpan().GetSeconds();

        if (elapsedTime < PresenceStatusStabilizationTime)
        {
            int remainTime = static_cast<int>(PresenceStatusStabilizationTime - elapsedTime);

            if (remainTime > wait)
            {
                NN_DETAIL_FRIENDS_INFO("[friends] UpdateUserPresence is pending state. (wait = %d -> %d)\n", wait, remainTime);
                wait = remainTime;
            }
        }
    }

    if (wait == 0)
    {
        SetRunnable(index, taskId);
    }
    else
    {
        m_Records[index].tasks[taskId].status = TaskStatus_Wait;
        m_Records[index].tasks[taskId].nextRunnableTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(wait));

        SetScheduleTimer();
    }
}

void BackgroundTaskManager::SetScheduleTimer() NN_NOEXCEPT
{
    nn::os::Tick nextRunnableTick = nn::os::Tick(INT64_MAX);
    bool hasWaitTask = false;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].uid == nn::account::InvalidUid)
        {
            continue;
        }

        for (int n = 0; n < NN_ARRAY_SIZE(m_Records[i].tasks); n++)
        {
            if (m_Records[i].tasks[n].status == TaskStatus_Wait)
            {
                if (m_Records[i].tasks[n].nextRunnableTick < nextRunnableTick)
                {
                    nextRunnableTick = m_Records[i].tasks[n].nextRunnableTick;
                    hasWaitTask = true;
                }
            }
        }
    }

    if (!hasWaitTask)
    {
        return;
    }

    if (m_IsSuspended)
    {
        if (nextRunnableTick < m_ResumeTick)
        {
            nextRunnableTick = m_ResumeTick;
        }
    }

    nn::TimeSpan interval = (nextRunnableTick - nn::os::GetSystemTick()).ToTimeSpan();

    if (interval > 0)
    {
        m_ScheduleEvent.StartOneShot(interval);
    }
    else
    {
        m_ScheduleEvent.Signal();
    }
}

bool BackgroundTaskManager::RescheduleUpdateUserPresenceTask(int index) NN_NOEXCEPT
{
    int64_t elapsedTime = (nn::os::GetSystemTick() - m_SleepAwakedTick).ToTimeSpan().GetSeconds();

    if (elapsedTime >= PresenceStatusStabilizationTime)
    {
        return false;
    }

    Wait(index, TaskId_UpdateUserPresence, static_cast<int>(PresenceStatusStabilizationTime - elapsedTime));

    return true;
}

void BackgroundTaskManager::SuspendTemporarily(int wait) NN_NOEXCEPT
{
    m_ResumeTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(wait));

    m_IsSuspended = true;
    SetScheduleTimer();
}

void BackgroundTaskManager::Resume() NN_NOEXCEPT
{
    m_RetryIntervalTableIndex = 0;

    if (m_IsSuspended)
    {
        m_IsSuspended = false;
        SetScheduleTimer();
    }
}

int BackgroundTaskManager::GetPriority(TaskId taskId) NN_NOEXCEPT
{
    // 大きい数字の方が高優先度となる。
    switch (taskId)
    {
    case TaskId_GetListSummary:
        return 90;
    case TaskId_SyncUser:
        return 85;
    case TaskId_SyncFriendList:
        return 80;
    case TaskId_SyncBlockedUserList:
        return 75;
    case TaskId_SendFacedFriendRequest:
        return 50;
    case TaskId_DownloadFriendProfileImage:
        return 25;
    case TaskId_UpdateUserPresence:
        return 100;
    case TaskId_GetFriendPresence:
        return 30;
    case TaskId_GetFriendRequestCount:
        return 40;
    default:
        return -1;
    }
}

}}}}}
