﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_Account.h>
#include <nn/friends/detail/service/friends_Common.h>
#include <nn/nsd/nsd_ApiForNasService.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    nn::os::SdkMutexType g_MutexForBuffer = NN_OS_SDK_MUTEX_INITIALIZER();

    NN_ALIGNAS(4096) Bit8 g_WorkBuffer[nn::account::RequiredBufferSizeForNintendoAccountAuthorizationRequestContext];

    nn::os::SdkMutexType g_MutexForCache = NN_OS_SDK_MUTEX_INITIALIZER();

    Account::NintendoAccountIdToken g_IdTokenCache = {};
    nn::account::NintendoAccountId g_CachedAccountId = {};
}

nn::Result Account::GetNetworkServiceAccountId(nn::account::NetworkServiceAccountId* outId,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountManager(&manager, uid));

    NN_RESULT_THROW_UNLESS(manager.GetNetworkServiceAccountId(outId).IsSuccess(), ResultNetworkServiceAccountNotLinked());

    NN_RESULT_SUCCESS;
}

nn::Result Account::GetDeviceAccountId(uint64_t* outId,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::account::GetBaasDeviceAccountId(outId, uid));

    NN_RESULT_SUCCESS;
}

nn::Result Account::EnsureNetworkServiceAccessToken(const nn::account::Uid& uid,
    const detail::service::util::Cancelable& cancelable) NN_NOEXCEPT
{
    nn::account::AsyncContext async;
    NN_RESULT_DO(nn::account::EnsureBaasAccessTokenCacheAsync(&async, uid));

    nn::os::SystemEvent completionEvent;
    NN_RESULT_DO(async.GetSystemEvent(&completionEvent));

    while (!completionEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(5)))
    {
        if (cancelable.IsCanceled())
        {
            async.Cancel();
            completionEvent.Wait();
        }
    }

    NN_RESULT_THROW_UNLESS(!cancelable.IsCanceled(), ResultCanceled());

    NN_RESULT_DO(async.GetResult());

    NN_RESULT_SUCCESS;
}

nn::Result Account::GetNetworkServiceAccessToken(NetworkServiceAccessToken* outToken,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    size_t size;
    NN_RESULT_DO(nn::account::LoadBaasAccessTokenCache(&size, outToken->value, sizeof (outToken->value), uid));

    NN_RESULT_SUCCESS;
}

bool Account::IsUserOpened(const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::Uid uids[nn::account::UserCountMax];
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListOpenUsers(&count, uids, NN_ARRAY_SIZE(uids)));

    for (int i = 0; i < count; i++)
    {
        if (uids[i] == uid)
        {
            return true;
        }
    }

    return false;
}

bool Account::IsNetworkServiceAccountAvailable(const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;

    if (nn::account::GetNetworkServiceAccountManager(&manager, uid).IsFailure())
    {
        return false;
    }

    bool isAvailable;

    if (manager.IsNetworkServiceAccountAvailable(&isAvailable).IsFailure())
    {
        return false;
    }

    return isAvailable;
}

bool Account::IsChild(const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;

    if (nn::account::GetNetworkServiceAccountManager(&manager, uid).IsFailure())
    {
        return true;
    }

    // 2KB あるのでスタックサイズに注意。
    Bit8 buffer[nn::account::RequiredBufferSizeForCachedNintendoAccountInfo] = {};

    nn::account::CachedNintendoAccountInfoForSystemService accountInfo;

    if (manager.LoadCachedNintendoAccountInfo(&accountInfo, buffer, sizeof (buffer)).IsFailure())
    {
        return true;
    }

    return accountInfo.IsChild();
}

nn::Result Account::GetNintendoAccountIdTokenForFriendCandidate(NintendoAccountIdToken* outToken,
    const nn::account::Uid& uid, const detail::service::util::Cancelable& cancelable) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountManager(&manager, uid));

    nn::account::NintendoAccountId accountId;
    NN_RESULT_DO(manager.GetNintendoAccountId(&accountId));

    {
        std::lock_guard<decltype (g_MutexForCache)> lock(g_MutexForCache);

        if (accountId == g_CachedAccountId && g_IdTokenCache.value[0])
        {
            std::memcpy(outToken->value, g_IdTokenCache.value, sizeof (g_IdTokenCache.value));
            NN_RESULT_SUCCESS;
        }
    }

    nn::nsd::NasServiceSetting setting;
    NN_RESULT_DO(nn::nsd::GetNasServiceSetting(&setting, nn::nsd::NasServiceNameOfNxFriends));

    nn::account::NintendoAccountAuthorizationRequestParameters params = {"openid"};

    nn::util::SNPrintf(params.state, sizeof (params.state), "%016llx", detail::service::util::GetRandom());
    nn::util::SNPrintf(params.nonce, sizeof (params.nonce), "%016llx", detail::service::util::GetRandom());

    {
        std::lock_guard<decltype (g_MutexForBuffer)> lock(g_MutexForBuffer);

        nn::account::NintendoAccountAuthorizationRequestContext context;

        NN_RESULT_DO(manager.CreateNintendoAccountAuthorizationRequest(&context,
            setting.clientId, setting.redirectUri.value, nn::nsd::NasServiceSetting::RedirectUri::Size,
            params, g_WorkBuffer, sizeof (g_WorkBuffer)));

        nn::os::SystemEvent completionEvent;
        NN_RESULT_DO(context.GetSystemEvent(&completionEvent));

        while (!completionEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(5)))
        {
            if (cancelable.IsCanceled())
            {
                context.Cancel();
                completionEvent.Wait();
            }
        }

        NN_RESULT_THROW_UNLESS(!cancelable.IsCanceled(), ResultCanceled());

        NN_RESULT_DO(context.GetResult());

        size_t size;
        NN_RESULT_DO(context.GetIdToken(&size, outToken->value, sizeof (outToken->value)));
    }

    {
        std::lock_guard<decltype (g_MutexForCache)> lock(g_MutexForCache);

        std::memcpy(g_IdTokenCache.value, outToken->value, sizeof (g_IdTokenCache.value));
        g_CachedAccountId = accountId;
    }

    NN_RESULT_SUCCESS;
}

void Account::ClearNintendoAccountIdTokenForFriendCandidate() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_MutexForCache)> lock(g_MutexForCache);

    g_IdTokenCache.value[0] = '\0';
}

}}}}}
