﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/friends_PresenceAccessor.h>
#include <nn/friends/friends_Result.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Utf8StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace friends { namespace detail {

nn::Result PresenceAccessor::AddKeyValue(const char* key, const char* value, char* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(value);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    int keyLength1 = nn::util::Strnlen(key, static_cast<int>(size));
    int valueLength1 = nn::util::Strnlen(value, static_cast<int>(size));

    NN_RESULT_THROW_UNLESS(keyLength1 < static_cast<int>(size), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(valueLength1 < static_cast<int>(size), ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(VerifyKey(key, keyLength1), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(VerifyValue(value, valueLength1), ResultInvalidArgument());

    // バッファの終端と残りサイズの計算、及び、キーの重複確認
    while (size > 0)
    {
        int keyLength2 = nn::util::Strnlen(buffer, static_cast<int>(size));

        NN_RESULT_THROW_UNLESS(keyLength2 < static_cast<int>(size), ResultInvalidArgument());

        if (keyLength2 == 0)
        {
            break;
        }

        NN_RESULT_THROW_UNLESS(keyLength1 != keyLength2 || nn::util::Strncmp(key, buffer, static_cast<int>(size)) != 0,
            ResultDuplicatedKey());

        int keySize = keyLength2 + 1;

        buffer += keySize;
        size -= keySize;

        NN_RESULT_THROW_UNLESS(size > 0, ResultInvalidArgument());

        int valueLength2 = nn::util::Strnlen(buffer, static_cast<int>(size));

        NN_RESULT_THROW_UNLESS(valueLength2 < static_cast<int>(size), ResultInvalidArgument());

        int valueSize = valueLength2 + 1;

        buffer += valueSize;
        size -= valueSize;
    }

    int keySize = keyLength1 + 1;
    int valueSize = valueLength1 + 1;

    NN_RESULT_THROW_UNLESS(keySize + valueSize <= static_cast<int>(size), ResultOutOfResource());

    std::memcpy(buffer, key, keySize);
    buffer += keySize;

    std::memcpy(buffer, value, valueSize);

    NN_RESULT_SUCCESS;
}

const char* PresenceAccessor::SearchKey(const char* key, const char* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    int keyLength1 = nn::util::Strnlen(key, static_cast<int>(size));

    if (keyLength1 == 0 || !(keyLength1 < static_cast<int>(size)))
    {
        return nullptr;
    }

    while (size > 0)
    {
        int keyLength2 = nn::util::Strnlen(buffer, static_cast<int>(size));

        if (!(keyLength2 < static_cast<int>(size)))
        {
            return nullptr;
        }

        if (keyLength2 == 0)
        {
            return nullptr;
        }

        bool isSameKey = (keyLength1 == keyLength2 && nn::util::Strncmp(key, buffer, static_cast<int>(size)) == 0);

        int keySize = keyLength2 + 1;

        buffer += keySize;
        size -= keySize;

        if (size == 0)
        {
            return nullptr;
        }

        if (isSameKey)
        {
            return buffer;
        }

        int valueLength = nn::util::Strnlen(buffer, static_cast<int>(size));

        if (!(valueLength < static_cast<int>(size)))
        {
            return nullptr;
        }

        int valueSize = valueLength + 1;

        buffer += valueSize;
        size -= valueSize;
    }

    return nullptr;
}

nn::Result PresenceAccessor::GetKeyValueList(int* outCount, KeyValue* outKeyValues, int count, const char* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outKeyValues);
    NN_SDK_REQUIRES(count > 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    int actualCount = 0;

    while (size > 0)
    {
        int keyLength = nn::util::Strnlen(buffer, static_cast<int>(size));

        NN_RESULT_THROW_UNLESS(keyLength < static_cast<int>(size), ResultInvalidArgument());

        if (keyLength == 0)
        {
            break;
        }

        int keySize = keyLength + 1;

        for (int i = 0; i < actualCount; i++)
        {
            // 重複は発生していないはず。
            NN_RESULT_THROW_UNLESS(nn::util::Strncmp(outKeyValues[i].key, buffer, keySize) != 0, ResultInvalidArgument());
        }

        outKeyValues[actualCount].key = buffer;
        outKeyValues[actualCount].value = buffer + keySize;

        NN_RESULT_THROW_UNLESS(VerifyKey(outKeyValues[actualCount].key, keyLength), ResultInvalidArgument());

        buffer += keySize;
        size -= keySize;

        NN_RESULT_THROW_UNLESS(size > 0, ResultInvalidArgument());

        int valueLength = nn::util::Strnlen(outKeyValues[actualCount].value, static_cast<int>(size));

        NN_RESULT_THROW_UNLESS(valueLength < static_cast<int>(size), ResultInvalidArgument());

        NN_RESULT_THROW_UNLESS(VerifyValue(outKeyValues[actualCount].value, valueLength), ResultInvalidArgument());

        int valueSize = valueLength + 1;

        buffer += valueSize;
        size -= valueSize;

        actualCount++;

        if (actualCount == count)
        {
            break;
        }
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

bool PresenceAccessor::VerifyKey(const char* key, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);

    if (length == 0)
    {
        return false;
    }

    const char* p = key;

    while (length > 0)
    {
        char c = *p;

        // [0-9a-zA-Z_]
        if (!((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c == '_')))
        {
            return false;
        }

        p++;
        length--;
    }

    return true;
}

bool PresenceAccessor::VerifyValue(const char* value, size_t length) NN_NOEXCEPT
{
    if (length == 0)
    {
        return true;
    }

    return nn::util::VerifyUtf8String(value, length);
}

}}}
